"""
Config Manager Module - Handles profile save/load and settings persistence

Profiles are stored as JSON files in the profiles/ directory.
"""

import json
import os
from dataclasses import dataclass, asdict, field
from typing import List, Optional
from pathlib import Path


@dataclass
class HotkeyConfig:
    """Configuration for a single hotkey"""
    key_name: Optional[str] = None  # Display name of the key (e.g., "F6")
    enabled: bool = True  # Whether this hotkey is enabled

    def to_dict(self) -> dict:
        return {"key": {"name": self.key_name} if self.key_name else None, "enabled": self.enabled}

    @classmethod
    def from_dict(cls, data: dict) -> 'HotkeyConfig':
        key_data = data.get("key")
        key_name = None
        if key_data:
            key_name = key_data.get("name") if isinstance(key_data, dict) else key_data
        return cls(key_name=key_name, enabled=data.get("enabled", True))


@dataclass
class HotkeysConfig:
    """Configuration for all hotkeys"""
    toggle: HotkeyConfig = field(default_factory=lambda: HotkeyConfig(key_name="F6"))
    hold: HotkeyConfig = field(default_factory=lambda: HotkeyConfig(enabled=False))
    focus_hold: HotkeyConfig = field(default_factory=lambda: HotkeyConfig(enabled=False))
    right_click_hold: HotkeyConfig = field(default_factory=lambda: HotkeyConfig(enabled=False))

    def to_dict(self) -> dict:
        return {
            "toggle": self.toggle.to_dict(),
            "hold": self.hold.to_dict(),
            "focus_hold": self.focus_hold.to_dict(),
            "right_click_hold": self.right_click_hold.to_dict(),
        }

    @classmethod
    def from_dict(cls, data: dict) -> 'HotkeysConfig':
        return cls(
            toggle=HotkeyConfig.from_dict(data.get("toggle", {})),
            hold=HotkeyConfig.from_dict(data.get("hold", {})),
            focus_hold=HotkeyConfig.from_dict(data.get("focus_hold", {})),
            right_click_hold=HotkeyConfig.from_dict(data.get("right_click_hold", {})),
        )


@dataclass
class ProfileData:
    """Data structure for a saved profile"""
    name: str = "Default"
    keys: List[str] = field(default_factory=list)
    spacebar_enabled: bool = False
    left_click_enabled: bool = False
    right_click_enabled: bool = False
    target_windows: str = ""
    pause_on_unfocus: bool = True
    hotkeys: HotkeysConfig = field(default_factory=HotkeysConfig)

    def to_dict(self) -> dict:
        """Convert to dictionary"""
        data = asdict(self)
        # Custom serialization for hotkeys
        data["hotkeys"] = self.hotkeys.to_dict()
        return data

    @classmethod
    def from_dict(cls, data: dict) -> 'ProfileData':
        """Create from dictionary"""
        hotkeys_data = data.get('hotkeys', {})
        hotkeys = HotkeysConfig.from_dict(hotkeys_data) if hotkeys_data else HotkeysConfig()

        return cls(
            name=data.get('name', 'Default'),
            keys=data.get('keys', []),
            spacebar_enabled=data.get('spacebar_enabled', False),
            left_click_enabled=data.get('left_click_enabled', False),
            right_click_enabled=data.get('right_click_enabled', False),
            target_windows=data.get('target_windows', ''),
            pause_on_unfocus=data.get('pause_on_unfocus', True),
            hotkeys=hotkeys,
        )


class ConfigManager:
    """
    Manages profile saving, loading, and deletion.

    Profiles are stored as individual JSON files in the profiles/ directory.
    """

    def __init__(self, profiles_dir: Optional[str] = None):
        """
        Initialize the config manager.

        Args:
            profiles_dir: Directory to store profiles. Defaults to ./profiles/
        """
        if profiles_dir:
            self.profiles_dir = Path(profiles_dir)
        else:
            # Default to profiles/ directory relative to this file's parent
            self.profiles_dir = Path(__file__).parent.parent / 'profiles'

        # Ensure profiles directory exists
        self.profiles_dir.mkdir(parents=True, exist_ok=True)

        # Create default profile if no profiles exist
        if not list(self.profiles_dir.glob('*.json')):
            self.save_profile(ProfileData())

    def _get_profile_path(self, name: str) -> Path:
        """Get the file path for a profile"""
        # Sanitize name for filename
        safe_name = "".join(c for c in name if c.isalnum() or c in (' ', '-', '_')).strip()
        safe_name = safe_name or 'Default'
        return self.profiles_dir / f"{safe_name}.json"

    def save_profile(self, profile: ProfileData) -> bool:
        """
        Save a profile to disk.

        Args:
            profile: ProfileData to save

        Returns:
            bool: True if successful
        """
        try:
            path = self._get_profile_path(profile.name)
            with open(path, 'w', encoding='utf-8') as f:
                json.dump(profile.to_dict(), f, indent=2)
            print(f"[ConfigManager] Saved profile: {profile.name}")
            return True
        except Exception as e:
            print(f"[ConfigManager] Error saving profile: {e}")
            return False

    def load_profile(self, name: str) -> Optional[ProfileData]:
        """
        Load a profile from disk.

        Args:
            name: Profile name to load

        Returns:
            ProfileData if found, None otherwise
        """
        try:
            path = self._get_profile_path(name)
            if not path.exists():
                print(f"[ConfigManager] Profile not found: {name}")
                return None

            with open(path, 'r', encoding='utf-8') as f:
                data = json.load(f)
            print(f"[ConfigManager] Loaded profile: {name}")
            return ProfileData.from_dict(data)
        except Exception as e:
            print(f"[ConfigManager] Error loading profile: {e}")
            return None

    def delete_profile(self, name: str) -> bool:
        """
        Delete a profile from disk.

        Args:
            name: Profile name to delete

        Returns:
            bool: True if successful
        """
        try:
            path = self._get_profile_path(name)
            if path.exists():
                path.unlink()
                print(f"[ConfigManager] Deleted profile: {name}")
                return True
            return False
        except Exception as e:
            print(f"[ConfigManager] Error deleting profile: {e}")
            return False

    def list_profiles(self) -> List[str]:
        """
        List all available profile names.

        Returns:
            List of profile names
        """
        profiles = []
        for path in self.profiles_dir.glob('*.json'):
            try:
                with open(path, 'r', encoding='utf-8') as f:
                    data = json.load(f)
                profiles.append(data.get('name', path.stem))
            except Exception:
                profiles.append(path.stem)
        return sorted(profiles)

    def profile_exists(self, name: str) -> bool:
        """Check if a profile exists"""
        return self._get_profile_path(name).exists()

    def get_default_profile(self) -> ProfileData:
        """Get the default profile, creating it if necessary"""
        profiles = self.list_profiles()
        if profiles:
            profile = self.load_profile(profiles[0])
            if profile:
                return profile
        return ProfileData()


# Singleton instance
_default_manager = None


def get_config_manager() -> ConfigManager:
    """Get the default config manager instance"""
    global _default_manager
    if _default_manager is None:
        _default_manager = ConfigManager()
    return _default_manager
