"""
Right Panel - Scrollable panel for profiles and controls

Contains:
- Profiles Panel (save/load/delete profiles)
- Focus & Timing Panel (target windows, pause on unfocus)
- Controls Panel (start/stop buttons)
"""

import customtkinter as ctk
from typing import Callable, Optional, List

import sys
import os
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))

from gui.theme import COLORS, FONTS, WIDGET_CONFIG
from gui.components.profiles_panel import ProfilesPanel
from gui.components.focus_timing import FocusTimingPanel
from gui.components.controls_panel import ControlsPanel


class RightPanel(ctk.CTkScrollableFrame):
    """
    Scrollable right panel containing profile and control widgets.

    Features:
    - Profile management (save/load/delete)
    - Window focus targeting
    - Start/Stop controls
    """

    def __init__(
        self,
        master,
        # Profile callbacks
        profiles: Optional[List[str]] = None,
        on_profile_select: Optional[Callable[[str], None]] = None,
        on_profile_save: Optional[Callable[[str], None]] = None,
        on_profile_delete: Optional[Callable[[str], None]] = None,
        # Focus timing callbacks
        on_focus_timing_change: Optional[Callable[[str, bool], None]] = None,
        # Control callbacks
        on_start: Optional[Callable[[], None]] = None,
        on_stop: Optional[Callable[[], None]] = None,
        **kwargs
    ):
        # Set default styling for scrollable frame
        super().__init__(
            master,
            fg_color=COLORS["bg_panel"],
            corner_radius=WIDGET_CONFIG["corner_radius_medium"],
            label_text="SETTINGS & CONTROLS",
            label_font=(FONTS["family"], FONTS["size_section"], FONTS["weight_bold"]),
            label_text_color=COLORS["accent_orange"],
            label_fg_color=COLORS["bg_panel"],
            scrollbar_button_color=COLORS["accent_orange"],
            scrollbar_button_hover_color=COLORS["accent_orange_hover"],
            **kwargs
        )

        # Configure internal grid
        self.grid_columnconfigure(0, weight=1)

        # Create widgets
        self._create_widgets(
            profiles or [],
            on_profile_select,
            on_profile_save,
            on_profile_delete,
            on_focus_timing_change,
            on_start,
            on_stop
        )

    def _create_widgets(
        self,
        profiles,
        on_profile_select,
        on_profile_save,
        on_profile_delete,
        on_focus_timing_change,
        on_start,
        on_stop
    ):
        """Create and layout child widgets"""
        row = 0

        # Profiles Panel
        self.profiles_panel = ProfilesPanel(
            self,
            profiles=profiles,
            on_profile_select=on_profile_select,
            on_save=on_profile_save,
            on_delete=on_profile_delete
        )
        self.profiles_panel.grid(row=row, column=0, sticky="ew", padx=5, pady=(5, 10))
        row += 1

        # Focus & Timing Panel
        self.focus_timing_panel = FocusTimingPanel(
            self,
            on_change=on_focus_timing_change
        )
        self.focus_timing_panel.grid(row=row, column=0, sticky="ew", padx=5, pady=5)
        row += 1

        # Controls Panel
        self.controls_panel = ControlsPanel(
            self,
            on_start=on_start,
            on_stop=on_stop
        )
        self.controls_panel.grid(row=row, column=0, sticky="ew", padx=5, pady=(5, 10))

    # =========================================================================
    # Public Methods - Profiles Panel
    # =========================================================================

    def set_profiles(self, profiles: List[str]):
        """Set available profiles list"""
        self.profiles_panel.set_profiles(profiles)

    def set_current_profile(self, name: str):
        """Set current profile selection"""
        self.profiles_panel.set_current_profile(name)

    def get_current_profile(self) -> str:
        """Get current profile name"""
        return self.profiles_panel.get_current_profile()

    # =========================================================================
    # Public Methods - Focus Timing Panel
    # =========================================================================

    def set_target_windows(self, windows: str):
        """Set target windows text"""
        self.focus_timing_panel.set_target_windows(windows)

    def set_pause_on_unfocus(self, enabled: bool):
        """Set pause on unfocus state"""
        self.focus_timing_panel.set_pause_on_unfocus(enabled)

    def get_focus_settings(self) -> tuple:
        """Get focus settings (target_windows, pause_on_unfocus)"""
        return self.focus_timing_panel.get_settings()

    # =========================================================================
    # Public Methods - Controls Panel
    # =========================================================================

    def set_running(self):
        """Set controls to running state"""
        self.controls_panel.set_running()

    def set_stopped(self):
        """Set controls to stopped state"""
        self.controls_panel.set_stopped()

    def set_paused(self):
        """Set controls to paused state"""
        self.controls_panel.set_paused()
