"""
Quick Actions Panel - Checkboxes for spacebar, left click, right click

SCARMONIT themed component for quick toggle actions.
"""

import customtkinter as ctk
from typing import Callable, Optional

import sys
import os
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))

from gui.theme import COLORS, FONTS, CHECKBOX_STYLES, WIDGET_CONFIG


class QuickActionsPanel(ctk.CTkFrame):
    """
    Quick actions panel with checkboxes for common actions.

    Layout:
    | QUICK ACTIONS
    ☑ Spacebar    ☑ Left Click    ☐ Right Click
    """

    def __init__(
        self,
        master,
        spacebar_enabled: bool = False,
        left_click_enabled: bool = False,
        right_click_enabled: bool = False,
        on_change: Optional[Callable[[bool, bool, bool], None]] = None,
        **kwargs
    ):
        """
        Initialize the quick actions panel.

        Args:
            master: Parent widget
            spacebar_enabled: Initial spacebar state
            left_click_enabled: Initial left click state
            right_click_enabled: Initial right click state
            on_change: Callback when any checkbox changes (spacebar, left, right)
        """
        super().__init__(
            master,
            fg_color=COLORS["bg_panel"],
            corner_radius=WIDGET_CONFIG["corner_radius_medium"],
            **kwargs
        )

        self.on_change = on_change

        # State variables
        self.spacebar_var = ctk.BooleanVar(value=spacebar_enabled)
        self.left_click_var = ctk.BooleanVar(value=left_click_enabled)
        self.right_click_var = ctk.BooleanVar(value=right_click_enabled)

        self._create_widgets()

    def _create_widgets(self):
        """Create and layout all widgets"""
        # Section header with orange bar
        header_frame = ctk.CTkFrame(self, fg_color="transparent")
        header_frame.pack(fill="x", padx=15, pady=(15, 10))

        # Orange bar
        orange_bar = ctk.CTkFrame(
            header_frame,
            width=3,
            height=20,
            fg_color=COLORS["section_bar"],
            corner_radius=0
        )
        orange_bar.pack(side="left", padx=(0, 8))

        # Section label
        header_label = ctk.CTkLabel(
            header_frame,
            text="QUICK ACTIONS",
            font=(FONTS["family"], FONTS["size_section"], FONTS["weight_bold"]),
            text_color=COLORS["text_primary"]
        )
        header_label.pack(side="left")

        # Checkboxes frame
        checkbox_frame = ctk.CTkFrame(self, fg_color="transparent")
        checkbox_frame.pack(fill="x", padx=15, pady=(0, 15))

        # Spacebar checkbox
        self.spacebar_checkbox = ctk.CTkCheckBox(
            checkbox_frame,
            text="Spacebar",
            variable=self.spacebar_var,
            command=self._on_checkbox_change,
            fg_color=CHECKBOX_STYLES["fg_color"],
            hover_color=CHECKBOX_STYLES["hover_color"],
            border_color=CHECKBOX_STYLES["border_color"],
            checkmark_color=CHECKBOX_STYLES["checkmark_color"],
            text_color=CHECKBOX_STYLES["text_color"],
            corner_radius=CHECKBOX_STYLES["corner_radius"],
            font=(FONTS["family"], FONTS["size_normal"])
        )
        self.spacebar_checkbox.pack(side="left", padx=(0, 20))

        # Left Click checkbox
        self.left_click_checkbox = ctk.CTkCheckBox(
            checkbox_frame,
            text="Left Click",
            variable=self.left_click_var,
            command=self._on_checkbox_change,
            fg_color=CHECKBOX_STYLES["fg_color"],
            hover_color=CHECKBOX_STYLES["hover_color"],
            border_color=CHECKBOX_STYLES["border_color"],
            checkmark_color=CHECKBOX_STYLES["checkmark_color"],
            text_color=CHECKBOX_STYLES["text_color"],
            corner_radius=CHECKBOX_STYLES["corner_radius"],
            font=(FONTS["family"], FONTS["size_normal"])
        )
        self.left_click_checkbox.pack(side="left", padx=(0, 20))

        # Right Click checkbox
        self.right_click_checkbox = ctk.CTkCheckBox(
            checkbox_frame,
            text="Right Click",
            variable=self.right_click_var,
            command=self._on_checkbox_change,
            fg_color=CHECKBOX_STYLES["fg_color"],
            hover_color=CHECKBOX_STYLES["hover_color"],
            border_color=CHECKBOX_STYLES["border_color"],
            checkmark_color=CHECKBOX_STYLES["checkmark_color"],
            text_color=CHECKBOX_STYLES["text_color"],
            corner_radius=CHECKBOX_STYLES["corner_radius"],
            font=(FONTS["family"], FONTS["size_normal"])
        )
        self.right_click_checkbox.pack(side="left")

    def _on_checkbox_change(self):
        """Handle checkbox state change"""
        if self.on_change:
            self.on_change(
                self.spacebar_var.get(),
                self.left_click_var.get(),
                self.right_click_var.get()
            )

    def set_states(
        self,
        spacebar: bool = None,
        left_click: bool = None,
        right_click: bool = None
    ):
        """Set checkbox states"""
        if spacebar is not None:
            self.spacebar_var.set(spacebar)
        if left_click is not None:
            self.left_click_var.set(left_click)
        if right_click is not None:
            self.right_click_var.set(right_click)

    def get_states(self) -> tuple:
        """Get current checkbox states"""
        return (
            self.spacebar_var.get(),
            self.left_click_var.get(),
            self.right_click_var.get()
        )

    def is_spacebar_enabled(self) -> bool:
        """Check if spacebar is enabled"""
        return self.spacebar_var.get()

    def is_left_click_enabled(self) -> bool:
        """Check if left click is enabled"""
        return self.left_click_var.get()

    def is_right_click_enabled(self) -> bool:
        """Check if right click is enabled"""
        return self.right_click_var.get()
