"""
Profiles Panel - Profile selection dropdown with Save/Delete buttons

SCARMONIT themed component for managing saved profiles.
"""

import customtkinter as ctk
from typing import Callable, List, Optional

import sys
import os
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))

from gui.theme import COLORS, FONTS, BUTTON_STYLES, DROPDOWN_STYLES, WIDGET_CONFIG


class ProfilesPanel(ctk.CTkFrame):
    """
    Profile management panel with dropdown selector and Save/Delete buttons.

    Layout:
    | PROFILES
    [Dropdown ▼]  [Save]  [Delete]
    """

    def __init__(
        self,
        master,
        profiles: List[str] = None,
        on_profile_select: Optional[Callable[[str], None]] = None,
        on_save: Optional[Callable[[str], None]] = None,
        on_delete: Optional[Callable[[str], None]] = None,
        **kwargs
    ):
        """
        Initialize the profiles panel.

        Args:
            master: Parent widget
            profiles: List of available profile names
            on_profile_select: Callback when profile is selected
            on_save: Callback when save button is clicked
            on_delete: Callback when delete button is clicked
        """
        super().__init__(
            master,
            fg_color=COLORS["bg_panel"],
            corner_radius=WIDGET_CONFIG["corner_radius_medium"],
            **kwargs
        )

        self.profiles = profiles or ["Default"]
        self.on_profile_select = on_profile_select
        self.on_save = on_save
        self.on_delete = on_delete

        self._create_widgets()

    def _create_widgets(self):
        """Create and layout all widgets"""
        # Configure grid
        self.grid_columnconfigure(0, weight=1)
        self.grid_columnconfigure(1, weight=0)
        self.grid_columnconfigure(2, weight=0)

        # Section header with orange bar
        header_frame = ctk.CTkFrame(self, fg_color="transparent")
        header_frame.grid(row=0, column=0, columnspan=3, sticky="w", padx=15, pady=(15, 10))

        # Orange bar
        orange_bar = ctk.CTkFrame(
            header_frame,
            width=3,
            height=20,
            fg_color=COLORS["section_bar"],
            corner_radius=0
        )
        orange_bar.pack(side="left", padx=(0, 8))

        # Section label
        header_label = ctk.CTkLabel(
            header_frame,
            text="PROFILES",
            font=(FONTS["family"], FONTS["size_section"], FONTS["weight_bold"]),
            text_color=COLORS["text_primary"]
        )
        header_label.pack(side="left")

        # Profile dropdown
        self.profile_dropdown = ctk.CTkOptionMenu(
            self,
            values=self.profiles,
            command=self._on_dropdown_change,
            width=150,
            height=32,
            fg_color=DROPDOWN_STYLES["fg_color"],
            button_color=DROPDOWN_STYLES["button_color"],
            button_hover_color=DROPDOWN_STYLES["button_hover_color"],
            dropdown_fg_color=DROPDOWN_STYLES["dropdown_fg_color"],
            dropdown_hover_color=DROPDOWN_STYLES["dropdown_hover_color"],
            text_color=DROPDOWN_STYLES["text_color"],
            corner_radius=DROPDOWN_STYLES["corner_radius"],
            font=(FONTS["family"], FONTS["size_normal"])
        )
        self.profile_dropdown.grid(row=1, column=0, sticky="w", padx=15, pady=(0, 15))

        # Save button
        self.save_button = ctk.CTkButton(
            self,
            text="Save",
            command=self._on_save_click,
            width=70,
            height=32,
            fg_color=BUTTON_STYLES["primary"]["fg_color"],
            hover_color=BUTTON_STYLES["primary"]["hover_color"],
            text_color=BUTTON_STYLES["primary"]["text_color"],
            corner_radius=BUTTON_STYLES["primary"]["corner_radius"],
            font=(FONTS["family"], FONTS["size_normal"], FONTS["weight_bold"])
        )
        self.save_button.grid(row=1, column=1, padx=(5, 5), pady=(0, 15))

        # Delete button
        self.delete_button = ctk.CTkButton(
            self,
            text="Delete",
            command=self._on_delete_click,
            width=70,
            height=32,
            fg_color=BUTTON_STYLES["danger"]["fg_color"],
            hover_color=BUTTON_STYLES["danger"]["hover_color"],
            text_color=BUTTON_STYLES["danger"]["text_color"],
            corner_radius=BUTTON_STYLES["danger"]["corner_radius"],
            font=(FONTS["family"], FONTS["size_normal"], FONTS["weight_bold"])
        )
        self.delete_button.grid(row=1, column=2, padx=(0, 15), pady=(0, 15))

    def _on_dropdown_change(self, choice: str):
        """Handle dropdown selection change"""
        if self.on_profile_select:
            self.on_profile_select(choice)

    def _on_save_click(self):
        """Handle save button click"""
        if self.on_save:
            current_profile = self.profile_dropdown.get()
            self.on_save(current_profile)

    def _on_delete_click(self):
        """Handle delete button click"""
        if self.on_delete:
            current_profile = self.profile_dropdown.get()
            self.on_delete(current_profile)

    def set_profiles(self, profiles: List[str]):
        """Update the list of available profiles"""
        self.profiles = profiles if profiles else ["Default"]
        self.profile_dropdown.configure(values=self.profiles)

    def set_current_profile(self, name: str):
        """Set the currently selected profile"""
        if name in self.profiles:
            self.profile_dropdown.set(name)

    def get_current_profile(self) -> str:
        """Get the currently selected profile name"""
        return self.profile_dropdown.get()

    def add_profile(self, name: str):
        """Add a new profile to the list"""
        if name not in self.profiles:
            self.profiles.append(name)
            self.profile_dropdown.configure(values=self.profiles)

    def remove_profile(self, name: str):
        """Remove a profile from the list"""
        if name in self.profiles and len(self.profiles) > 1:
            self.profiles.remove(name)
            self.profile_dropdown.configure(values=self.profiles)
            self.profile_dropdown.set(self.profiles[0])
