"""
Left Panel - Scrollable panel for key configuration

Contains:
- Hotkeys Card (capture hotkeys)
- Key Config Panel (keys to spam)
- Quick Actions Panel (spacebar, left/right click)
"""

import customtkinter as ctk
from typing import Callable, Optional, List

import sys
import os
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))

from gui.theme import COLORS, FONTS, WIDGET_CONFIG
from gui.components.compact_hotkeys_card import CompactHotkeysCard
from gui.components.key_config_panel import KeyConfigPanel
from gui.components.quick_actions import QuickActionsPanel


class LeftPanel(ctk.CTkScrollableFrame):
    """
    Scrollable left panel containing key configuration widgets.

    Features:
    - Hotkey capture system
    - Key list configuration
    - Quick action checkboxes
    """

    def __init__(
        self,
        master,
        # Hotkey callbacks
        on_toggle_capture_start: Optional[Callable[[], None]] = None,
        on_hold_capture_start: Optional[Callable[[], None]] = None,
        on_focus_capture_start: Optional[Callable[[], None]] = None,
        on_rclick_capture_start: Optional[Callable[[], None]] = None,
        on_hold_enabled_change: Optional[Callable[[bool], None]] = None,
        on_focus_enabled_change: Optional[Callable[[bool], None]] = None,
        on_rclick_enabled_change: Optional[Callable[[bool], None]] = None,
        # Key config callbacks
        on_keys_change: Optional[Callable[[List[str]], None]] = None,
        # Quick actions callbacks
        on_quick_actions_change: Optional[Callable[[bool, bool, bool], None]] = None,
        **kwargs
    ):
        # Set default styling for scrollable frame
        super().__init__(
            master,
            fg_color=COLORS["bg_panel"],
            corner_radius=WIDGET_CONFIG["corner_radius_medium"],
            label_text="KEY CONFIGURATION",
            label_font=(FONTS["family"], FONTS["size_section"], FONTS["weight_bold"]),
            label_text_color=COLORS["accent_orange"],
            label_fg_color=COLORS["bg_panel"],
            scrollbar_button_color=COLORS["accent_orange"],
            scrollbar_button_hover_color=COLORS["accent_orange_hover"],
            **kwargs
        )

        # Configure internal grid
        self.grid_columnconfigure(0, weight=1)

        # Create widgets
        self._create_widgets(
            on_toggle_capture_start,
            on_hold_capture_start,
            on_focus_capture_start,
            on_rclick_capture_start,
            on_hold_enabled_change,
            on_focus_enabled_change,
            on_rclick_enabled_change,
            on_keys_change,
            on_quick_actions_change
        )

    def _create_widgets(
        self,
        on_toggle_capture_start,
        on_hold_capture_start,
        on_focus_capture_start,
        on_rclick_capture_start,
        on_hold_enabled_change,
        on_focus_enabled_change,
        on_rclick_enabled_change,
        on_keys_change,
        on_quick_actions_change
    ):
        """Create and layout child widgets"""
        row = 0

        # Hotkeys Card
        self.hotkeys_card = CompactHotkeysCard(
            self,
            on_toggle_capture_start=on_toggle_capture_start,
            on_hold_capture_start=on_hold_capture_start,
            on_focus_capture_start=on_focus_capture_start,
            on_rclick_capture_start=on_rclick_capture_start,
            on_hold_enabled_change=on_hold_enabled_change,
            on_focus_enabled_change=on_focus_enabled_change,
            on_rclick_enabled_change=on_rclick_enabled_change
        )
        self.hotkeys_card.grid(row=row, column=0, sticky="ew", padx=5, pady=(5, 10))
        row += 1

        # Key Configuration Panel
        self.key_config_panel = KeyConfigPanel(
            self,
            on_keys_change=on_keys_change
        )
        self.key_config_panel.grid(row=row, column=0, sticky="ew", padx=5, pady=5)
        row += 1

        # Quick Actions Panel
        self.quick_actions_panel = QuickActionsPanel(
            self,
            on_change=on_quick_actions_change
        )
        self.quick_actions_panel.grid(row=row, column=0, sticky="ew", padx=5, pady=(5, 10))

    # =========================================================================
    # Public Methods - Hotkeys Card
    # =========================================================================

    def set_hotkeys_config(self, config: dict):
        """Set hotkeys configuration"""
        self.hotkeys_card.set_config(config)

    def get_hotkeys_config(self) -> dict:
        """Get hotkeys configuration"""
        return self.hotkeys_card.get_config()

    def set_toggle_key(self, name: Optional[str]):
        """Set toggle key display"""
        self.hotkeys_card.set_toggle_key(name)

    def set_hold_key(self, name: Optional[str]):
        """Set hold key display"""
        self.hotkeys_card.set_hold_key(name)

    def set_focus_key(self, name: Optional[str]):
        """Set focus key display"""
        self.hotkeys_card.set_focus_key(name)

    def set_rclick_key(self, name: Optional[str]):
        """Set r-hold key display"""
        self.hotkeys_card.set_rclick_key(name)

    # =========================================================================
    # Public Methods - Key Config Panel
    # =========================================================================

    def set_keys(self, keys: List[str]):
        """Set the keys list"""
        self.key_config_panel.set_keys(keys)

    def get_keys(self) -> List[str]:
        """Get the keys list"""
        return self.key_config_panel.get_keys()

    # =========================================================================
    # Public Methods - Quick Actions Panel
    # =========================================================================

    def set_quick_actions(self, spacebar: bool, left_click: bool, right_click: bool):
        """Set quick action states"""
        self.quick_actions_panel.set_states(spacebar, left_click, right_click)

    def get_quick_actions(self) -> tuple:
        """Get quick action states (spacebar, left_click, right_click)"""
        return self.quick_actions_panel.get_states()
