"""
Key Config Panel - Add/remove keys to spam

SCARMONIT themed component for configuring which keys to spam.
Supports capturing both keyboard keys and mouse buttons.
"""

import customtkinter as ctk
from typing import Callable, List, Optional
import threading

import sys
import os
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))

from gui.theme import COLORS, FONTS, BUTTON_STYLES, INPUT_STYLES, WIDGET_CONFIG
from core.captured_key import CapturedKey
from pynput import keyboard, mouse


class KeyConfigPanel(ctk.CTkFrame):
    """
    Key configuration panel with entry field and scrollable key list.
    Supports capturing both keyboard keys and mouse buttons.

    Layout:
    | KEYS TO SPAM
    [Enter key...         ] [+] [Capture]
    [Scrollable list of keys with X buttons]
    """

    def __init__(
        self,
        master,
        keys: List[str] = None,
        on_keys_change: Optional[Callable[[List[str]], None]] = None,
        **kwargs
    ):
        """
        Initialize the key config panel.

        Args:
            master: Parent widget
            keys: Initial list of keys
            on_keys_change: Callback when keys list changes
        """
        super().__init__(
            master,
            fg_color=COLORS["bg_panel"],
            corner_radius=WIDGET_CONFIG["corner_radius_medium"],
            **kwargs
        )

        self.keys = keys or []
        self.on_keys_change = on_keys_change
        self._key_widgets = {}  # Track key widgets for removal

        # Capture mode state
        self._capturing = False
        self._keyboard_listener: Optional[keyboard.Listener] = None
        self._mouse_listener: Optional[mouse.Listener] = None

        self._create_widgets()
        self._populate_keys()

    def _create_widgets(self):
        """Create and layout all widgets"""
        # Configure grid
        self.grid_columnconfigure(0, weight=1)

        # Section header with orange bar
        header_frame = ctk.CTkFrame(self, fg_color="transparent")
        header_frame.grid(row=0, column=0, sticky="w", padx=10, pady=(8, 5))

        # Orange bar
        orange_bar = ctk.CTkFrame(
            header_frame,
            width=3,
            height=16,
            fg_color=COLORS["section_bar"],
            corner_radius=0
        )
        orange_bar.pack(side="left", padx=(0, 6))

        # Section label
        header_label = ctk.CTkLabel(
            header_frame,
            text="KEYS TO SPAM",
            font=(FONTS["family"], FONTS["size_small"], FONTS["weight_bold"]),
            text_color=COLORS["text_primary"]
        )
        header_label.pack(side="left")

        # Input row frame
        input_frame = ctk.CTkFrame(self, fg_color="transparent")
        input_frame.grid(row=1, column=0, sticky="ew", padx=10, pady=(0, 5))
        input_frame.grid_columnconfigure(0, weight=1)

        # Key entry
        self.key_entry = ctk.CTkEntry(
            input_frame,
            placeholder_text="Enter key...",
            height=28,
            fg_color=INPUT_STYLES["fg_color"],
            border_color=INPUT_STYLES["border_color"],
            text_color=INPUT_STYLES["text_color"],
            placeholder_text_color=INPUT_STYLES["placeholder_text_color"],
            corner_radius=INPUT_STYLES["corner_radius"],
            border_width=INPUT_STYLES["border_width"],
            font=(FONTS["family"], FONTS["size_small"])
        )
        self.key_entry.grid(row=0, column=0, sticky="ew", padx=(0, 5))
        self.key_entry.bind("<Return>", lambda e: self._add_key())

        # Add button
        self.add_button = ctk.CTkButton(
            input_frame,
            text="+",
            command=self._add_key,
            width=28,
            height=28,
            fg_color=BUTTON_STYLES["primary"]["fg_color"],
            hover_color=BUTTON_STYLES["primary"]["hover_color"],
            text_color=BUTTON_STYLES["primary"]["text_color"],
            corner_radius=BUTTON_STYLES["primary"]["corner_radius"],
            font=(FONTS["family"], FONTS["size_normal"], FONTS["weight_bold"])
        )
        self.add_button.grid(row=0, column=1, padx=(0, 3))

        # Capture button - for capturing keyboard/mouse input
        self.capture_button = ctk.CTkButton(
            input_frame,
            text="⌨",
            command=self._toggle_capture,
            width=28,
            height=28,
            fg_color=COLORS["bg_hover"],
            hover_color=COLORS["accent_orange"],
            text_color=COLORS["text_primary"],
            corner_radius=BUTTON_STYLES["primary"]["corner_radius"],
            font=(FONTS["family"], FONTS["size_normal"])
        )
        self.capture_button.grid(row=0, column=2)

        # Scrollable frame for keys (minimal height - just fits key tags)
        self.keys_frame = ctk.CTkScrollableFrame(
            self,
            height=20,
            fg_color=COLORS["bg_input"],
            corner_radius=WIDGET_CONFIG["corner_radius_small"]
        )
        self.keys_frame.grid(row=2, column=0, sticky="ew", padx=10, pady=(0, 5))

    def _populate_keys(self):
        """Populate the keys frame with current keys"""
        # Clear existing widgets
        for widget in self.keys_frame.winfo_children():
            widget.destroy()
        self._key_widgets.clear()

        # Add key tags
        for i, key in enumerate(self.keys):
            self._add_key_widget(key, i)

    def _add_key_widget(self, key: str, index: int):
        """Add a key tag widget"""
        key_frame = ctk.CTkFrame(
            self.keys_frame,
            fg_color=COLORS["bg_hover"],
            corner_radius=WIDGET_CONFIG["corner_radius_small"]
        )
        key_frame.pack(side="left", padx=3, pady=3)

        # Key label
        key_label = ctk.CTkLabel(
            key_frame,
            text=key.upper(),
            font=(FONTS["family"], FONTS["size_small"], FONTS["weight_bold"]),
            text_color=COLORS["text_primary"]
        )
        key_label.pack(side="left", padx=(10, 5), pady=5)

        # Remove button (×)
        remove_btn = ctk.CTkButton(
            key_frame,
            text="×",
            width=20,
            height=20,
            fg_color="transparent",
            hover_color=COLORS["accent_red"],
            text_color=COLORS["text_muted"],
            corner_radius=WIDGET_CONFIG["corner_radius_small"],
            font=(FONTS["family"], FONTS["size_normal"]),
            command=lambda k=key: self._remove_key(k)
        )
        remove_btn.pack(side="left", padx=(0, 5), pady=5)

        self._key_widgets[key] = key_frame

    def _add_key(self):
        """Add a new key from the entry field"""
        key = self.key_entry.get().strip().lower()
        if key and key not in self.keys:
            self.keys.append(key)
            self._add_key_widget(key, len(self.keys) - 1)
            self.key_entry.delete(0, "end")
            self._notify_change()

    def _remove_key(self, key: str):
        """Remove a key from the list"""
        if key in self.keys:
            self.keys.remove(key)
            if key in self._key_widgets:
                self._key_widgets[key].destroy()
                del self._key_widgets[key]
            self._notify_change()

    def _notify_change(self):
        """Notify callback of keys change"""
        if self.on_keys_change:
            self.on_keys_change(self.keys.copy())

    def set_keys(self, keys: List[str]):
        """Set the list of keys"""
        self.keys = keys.copy() if keys else []
        self._populate_keys()

    def get_keys(self) -> List[str]:
        """Get the current list of keys"""
        return self.keys.copy()

    def clear_keys(self):
        """Clear all keys"""
        self.keys.clear()
        self._populate_keys()
        self._notify_change()

    def _toggle_capture(self):
        """Toggle capture mode for keyboard/mouse input"""
        if self._capturing:
            self._stop_capture()
        else:
            self._start_capture()

    def _start_capture(self):
        """Start capturing keyboard and mouse input"""
        if self._capturing:
            return

        self._capturing = True

        # Update button appearance
        self.capture_button.configure(
            fg_color=COLORS["accent_orange"],
            text="..."
        )

        # Update entry placeholder
        self.key_entry.configure(placeholder_text="Press any key or mouse button...")

        # Start keyboard listener
        self._keyboard_listener = keyboard.Listener(
            on_press=self._on_key_capture
        )
        self._keyboard_listener.start()

        # Start mouse listener
        self._mouse_listener = mouse.Listener(
            on_click=self._on_mouse_capture
        )
        self._mouse_listener.start()

    def _stop_capture(self):
        """Stop capturing input"""
        if not self._capturing:
            return

        self._capturing = False

        # Stop listeners
        if self._keyboard_listener:
            self._keyboard_listener.stop()
            self._keyboard_listener = None

        if self._mouse_listener:
            self._mouse_listener.stop()
            self._mouse_listener = None

        # Restore button appearance
        self.capture_button.configure(
            fg_color=COLORS["bg_hover"],
            text="⌨"
        )

        # Restore entry placeholder
        self.key_entry.configure(placeholder_text="Enter key...")

    def _on_key_capture(self, key):
        """Handle captured keyboard key"""
        if not self._capturing:
            return

        # Create CapturedKey from pynput key
        captured = CapturedKey.from_pynput_key(key)
        key_name = captured.name.lower()

        # Schedule GUI update on main thread
        self.after(0, lambda: self._add_captured_key(key_name))

    def _on_mouse_capture(self, x, y, button, pressed):
        """Handle captured mouse button"""
        if not self._capturing or not pressed:
            return

        # Create CapturedKey from mouse button
        captured = CapturedKey.from_mouse_button(button)
        key_name = captured.name.lower()

        # Schedule GUI update on main thread
        self.after(0, lambda: self._add_captured_key(key_name))

    def _add_captured_key(self, key_name: str):
        """Add a captured key to the list and stop capture mode"""
        # Stop capture mode first
        self._stop_capture()

        # Add key if not duplicate
        if key_name and key_name not in self.keys:
            self.keys.append(key_name)
            self._add_key_widget(key_name, len(self.keys) - 1)
            self._notify_change()

    def destroy(self):
        """Clean up resources when widget is destroyed"""
        self._stop_capture()
        super().destroy()
