"""
Focus & Timing Panel - Target windows and pause on unfocus settings

SCARMONIT themed component for window focus configuration.
"""

import customtkinter as ctk
from typing import Callable, Optional

import sys
import os
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))

from gui.theme import COLORS, FONTS, INPUT_STYLES, CHECKBOX_STYLES, WIDGET_CONFIG


class FocusTimingPanel(ctk.CTkFrame):
    """
    Focus and timing panel with target windows input and pause checkbox.

    Layout:
    | FOCUS & TIMING
    Target windows (comma-separated):
    [Diablo IV                          ]
    ☑ Pause when game window unfocused
    """

    def __init__(
        self,
        master,
        target_windows: str = "",
        pause_on_unfocus: bool = True,
        on_change: Optional[Callable[[str, bool], None]] = None,
        **kwargs
    ):
        """
        Initialize the focus timing panel.

        Args:
            master: Parent widget
            target_windows: Initial target windows string
            pause_on_unfocus: Initial pause on unfocus state
            on_change: Callback when settings change (target_windows, pause_on_unfocus)
        """
        super().__init__(
            master,
            fg_color=COLORS["bg_panel"],
            corner_radius=WIDGET_CONFIG["corner_radius_medium"],
            **kwargs
        )

        self.on_change = on_change
        self.pause_var = ctk.BooleanVar(value=pause_on_unfocus)

        self._create_widgets()
        self._set_initial_values(target_windows)

    def _create_widgets(self):
        """Create and layout all widgets"""
        # Configure grid
        self.grid_columnconfigure(0, weight=1)

        # Section header with orange bar
        header_frame = ctk.CTkFrame(self, fg_color="transparent")
        header_frame.grid(row=0, column=0, sticky="w", padx=15, pady=(15, 10))

        # Orange bar
        orange_bar = ctk.CTkFrame(
            header_frame,
            width=3,
            height=20,
            fg_color=COLORS["section_bar"],
            corner_radius=0
        )
        orange_bar.pack(side="left", padx=(0, 8))

        # Section label
        header_label = ctk.CTkLabel(
            header_frame,
            text="FOCUS & TIMING",
            font=(FONTS["family"], FONTS["size_section"], FONTS["weight_bold"]),
            text_color=COLORS["text_primary"]
        )
        header_label.pack(side="left")

        # Description label
        desc_label = ctk.CTkLabel(
            self,
            text="Target windows (comma-separated):",
            font=(FONTS["family"], FONTS["size_small"]),
            text_color=COLORS["text_muted"]
        )
        desc_label.grid(row=1, column=0, sticky="w", padx=15, pady=(0, 5))

        # Target windows entry
        self.target_entry = ctk.CTkEntry(
            self,
            placeholder_text="Diablo IV",
            height=35,
            fg_color=INPUT_STYLES["fg_color"],
            border_color=INPUT_STYLES["border_color"],
            text_color=INPUT_STYLES["text_color"],
            placeholder_text_color=INPUT_STYLES["placeholder_text_color"],
            corner_radius=INPUT_STYLES["corner_radius"],
            border_width=INPUT_STYLES["border_width"],
            font=(FONTS["family"], FONTS["size_normal"])
        )
        self.target_entry.grid(row=2, column=0, sticky="ew", padx=15, pady=(0, 10))
        self.target_entry.bind("<FocusOut>", lambda e: self._on_entry_change())
        self.target_entry.bind("<Return>", lambda e: self._on_entry_change())

        # Pause on unfocus checkbox
        self.pause_checkbox = ctk.CTkCheckBox(
            self,
            text="Pause when game window unfocused",
            variable=self.pause_var,
            command=self._on_checkbox_change,
            fg_color=CHECKBOX_STYLES["fg_color"],
            hover_color=CHECKBOX_STYLES["hover_color"],
            border_color=CHECKBOX_STYLES["border_color"],
            checkmark_color=CHECKBOX_STYLES["checkmark_color"],
            text_color=CHECKBOX_STYLES["text_color"],
            corner_radius=CHECKBOX_STYLES["corner_radius"],
            font=(FONTS["family"], FONTS["size_normal"])
        )
        self.pause_checkbox.grid(row=3, column=0, sticky="w", padx=15, pady=(0, 15))

    def _set_initial_values(self, target_windows: str):
        """Set initial entry value"""
        if target_windows:
            self.target_entry.insert(0, target_windows)

    def _on_entry_change(self):
        """Handle target windows entry change"""
        self._notify_change()

    def _on_checkbox_change(self):
        """Handle pause checkbox change"""
        self._notify_change()

    def _notify_change(self):
        """Notify callback of settings change"""
        if self.on_change:
            self.on_change(
                self.target_entry.get(),
                self.pause_var.get()
            )

    def set_target_windows(self, windows: str):
        """Set target windows value"""
        self.target_entry.delete(0, "end")
        if windows:
            self.target_entry.insert(0, windows)

    def set_pause_on_unfocus(self, enabled: bool):
        """Set pause on unfocus state"""
        self.pause_var.set(enabled)

    def get_target_windows(self) -> str:
        """Get target windows value"""
        return self.target_entry.get()

    def get_pause_on_unfocus(self) -> bool:
        """Get pause on unfocus state"""
        return self.pause_var.get()

    def get_settings(self) -> tuple:
        """Get all settings as tuple"""
        return (self.get_target_windows(), self.get_pause_on_unfocus())
