"""
Controls Panel - Start/Stop buttons and status display

SCARMONIT themed component for spammer controls.
"""

import customtkinter as ctk
from typing import Callable, Optional

import sys
import os
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))

from gui.theme import COLORS, FONTS, WIDGET_CONFIG


class ControlsPanel(ctk.CTkFrame):
    """
    Controls panel with Start/Stop buttons and status indicator.

    Layout:
    | CONTROLS
        [▶ START (F6)]     [■ STOP (F7)]

    Status: Stopped
    """

    def __init__(
        self,
        master,
        on_start: Optional[Callable[[], None]] = None,
        on_stop: Optional[Callable[[], None]] = None,
        **kwargs
    ):
        """
        Initialize the controls panel.

        Args:
            master: Parent widget
            on_start: Callback when start button is clicked
            on_stop: Callback when stop button is clicked
        """
        super().__init__(
            master,
            fg_color=COLORS["bg_panel"],
            corner_radius=WIDGET_CONFIG["corner_radius_medium"],
            **kwargs
        )

        self.on_start = on_start
        self.on_stop = on_stop

        self._create_widgets()

    def _create_widgets(self):
        """Create and layout all widgets"""
        # Configure grid
        self.grid_columnconfigure(0, weight=1)

        # Section header with orange bar
        header_frame = ctk.CTkFrame(self, fg_color="transparent")
        header_frame.grid(row=0, column=0, sticky="w", padx=15, pady=(15, 10))

        # Orange bar
        orange_bar = ctk.CTkFrame(
            header_frame,
            width=3,
            height=20,
            fg_color=COLORS["section_bar"],
            corner_radius=0
        )
        orange_bar.pack(side="left", padx=(0, 8))

        # Section label
        header_label = ctk.CTkLabel(
            header_frame,
            text="CONTROLS",
            font=(FONTS["family"], FONTS["size_section"], FONTS["weight_bold"]),
            text_color=COLORS["text_primary"]
        )
        header_label.pack(side="left")

        # Buttons frame
        buttons_frame = ctk.CTkFrame(self, fg_color="transparent")
        buttons_frame.grid(row=1, column=0, pady=(0, 10))

        # Start button
        self.start_button = ctk.CTkButton(
            buttons_frame,
            text="▶ START (F6)",
            command=self._on_start_click,
            width=140,
            height=40,
            fg_color=COLORS["accent_green"],
            hover_color="#2ecc71",
            text_color=COLORS["text_primary"],
            corner_radius=WIDGET_CONFIG["corner_radius_medium"],
            font=(FONTS["family"], FONTS["size_normal"], FONTS["weight_bold"])
        )
        self.start_button.pack(side="left", padx=(0, 15))

        # Stop button
        self.stop_button = ctk.CTkButton(
            buttons_frame,
            text="■ STOP (F7)",
            command=self._on_stop_click,
            width=140,
            height=40,
            fg_color=COLORS["accent_red"],
            hover_color=COLORS["accent_red_hover"],
            text_color=COLORS["text_primary"],
            corner_radius=WIDGET_CONFIG["corner_radius_medium"],
            font=(FONTS["family"], FONTS["size_normal"], FONTS["weight_bold"])
        )
        self.stop_button.pack(side="left")

        # Status label
        self.status_label = ctk.CTkLabel(
            self,
            text="Status: Stopped",
            font=(FONTS["family"], FONTS["size_normal"]),
            text_color=COLORS["text_muted"]
        )
        self.status_label.grid(row=2, column=0, pady=(0, 15))

    def _on_start_click(self):
        """Handle start button click"""
        if self.on_start:
            self.on_start()

    def _on_stop_click(self):
        """Handle stop button click"""
        if self.on_stop:
            self.on_stop()

    def set_status(self, status: str, color: str = None):
        """
        Update the status display.

        Args:
            status: Status text to display
            color: Optional text color override
        """
        self.status_label.configure(text=f"Status: {status}")
        if color:
            self.status_label.configure(text_color=color)

    def set_running(self):
        """Set status to running"""
        self.set_status("Running", COLORS["status_running"])
        self.start_button.configure(state="disabled")
        self.stop_button.configure(state="normal")

    def set_stopped(self):
        """Set status to stopped"""
        self.set_status("Stopped", COLORS["status_stopped"])
        self.start_button.configure(state="normal")
        self.stop_button.configure(state="disabled")

    def set_paused(self):
        """Set status to paused (window unfocused)"""
        self.set_status("Paused (window unfocused)", COLORS["status_paused"])

    def enable_buttons(self, start: bool = True, stop: bool = True):
        """Enable/disable buttons"""
        self.start_button.configure(state="normal" if start else "disabled")
        self.stop_button.configure(state="normal" if stop else "disabled")
