"""
Compact Hotkeys Card - Horizontal row of 4 hotkey slots

SCARMONIT themed component for hotkey configuration with:
- Toggle: Press to start/stop
- Hold: Hold to spam, release to stop
- Focus: Hold to spam specific key only
- R-Hold: Hold to spam right-click only
"""

import customtkinter as ctk
from typing import Callable, Optional

import sys
import os
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.dirname(os.path.abspath(__file__)))))

from gui.theme import COLORS, FONTS, WIDGET_CONFIG


class CompactHotkeysCard(ctk.CTkFrame):
    """
    Compact horizontal hotkey configuration card.

    Layout:
    | HOTKEYS                                              |
    | Toggle    | Hold      | Focus     | R-Hold          |
    | [  F6  ]  | [  —  ]   | [  —  ]   | [  —  ]        |
    |           | [✓] Enable| [✓] Enable| [✓] Enable     |
    """

    # Default empty key display
    EMPTY_KEY = "—"
    CAPTURING_TEXT = "..."

    def __init__(
        self,
        master,
        on_toggle_capture_start: Optional[Callable[[], None]] = None,
        on_hold_capture_start: Optional[Callable[[], None]] = None,
        on_focus_capture_start: Optional[Callable[[], None]] = None,
        on_rclick_capture_start: Optional[Callable[[], None]] = None,
        on_hold_enabled_change: Optional[Callable[[bool], None]] = None,
        on_focus_enabled_change: Optional[Callable[[bool], None]] = None,
        on_rclick_enabled_change: Optional[Callable[[bool], None]] = None,
        **kwargs
    ):
        """
        Initialize the compact hotkeys card.

        Args:
            master: Parent widget
            on_toggle_capture_start: Callback when toggle capture starts
            on_hold_capture_start: Callback when hold capture starts
            on_focus_capture_start: Callback when focus capture starts
            on_rclick_capture_start: Callback when r-hold capture starts
            on_hold_enabled_change: Callback when hold enabled changes
            on_focus_enabled_change: Callback when focus enabled changes
            on_rclick_enabled_change: Callback when r-hold enabled changes
        """
        super().__init__(
            master,
            fg_color=COLORS["bg_panel"],
            corner_radius=WIDGET_CONFIG["corner_radius_medium"],
            **kwargs
        )

        # Store callbacks
        self._on_toggle_capture = on_toggle_capture_start
        self._on_hold_capture = on_hold_capture_start
        self._on_focus_capture = on_focus_capture_start
        self._on_rclick_capture = on_rclick_capture_start
        self._on_hold_enabled = on_hold_enabled_change
        self._on_focus_enabled = on_focus_enabled_change
        self._on_rclick_enabled = on_rclick_enabled_change

        # Enable state variables
        self._hold_enabled_var = ctk.BooleanVar(value=False)
        self._focus_enabled_var = ctk.BooleanVar(value=False)
        self._rclick_enabled_var = ctk.BooleanVar(value=False)

        # Key button references
        self._toggle_button: Optional[ctk.CTkButton] = None
        self._hold_button: Optional[ctk.CTkButton] = None
        self._focus_button: Optional[ctk.CTkButton] = None
        self._rclick_button: Optional[ctk.CTkButton] = None

        self._create_widgets()

    def _create_key_button(self, parent, text: str, command: Callable) -> ctk.CTkButton:
        """Create a styled key button"""
        return ctk.CTkButton(
            parent,
            text=text,
            command=command,
            width=70,
            height=35,
            fg_color=COLORS["bg_dark"],
            hover_color=COLORS["bg_hover"],
            border_color=COLORS["accent_orange"],
            border_width=1,
            text_color=COLORS["accent_orange"],
            corner_radius=WIDGET_CONFIG["corner_radius_small"],
            font=(FONTS["family"], FONTS["size_normal"], FONTS["weight_bold"])
        )

    def _create_widgets(self):
        """Create and layout all widgets"""
        # Configure grid - 4 equal columns
        for i in range(4):
            self.grid_columnconfigure(i, weight=1)

        # Section header with orange bar
        header_frame = ctk.CTkFrame(self, fg_color="transparent")
        header_frame.grid(row=0, column=0, columnspan=4, sticky="w", padx=15, pady=(15, 10))

        # Orange bar
        orange_bar = ctk.CTkFrame(
            header_frame,
            width=3,
            height=20,
            fg_color=COLORS["section_bar"],
            corner_radius=0
        )
        orange_bar.pack(side="left", padx=(0, 8))

        # Section label
        header_label = ctk.CTkLabel(
            header_frame,
            text="HOTKEYS",
            font=(FONTS["family"], FONTS["size_section"], FONTS["weight_bold"]),
            text_color=COLORS["text_primary"]
        )
        header_label.pack(side="left")

        # Create 4 columns
        self._create_toggle_column(0)
        self._create_hold_column(1)
        self._create_focus_column(2)
        self._create_rclick_column(3)

    def _create_toggle_column(self, col: int):
        """Create the Toggle hotkey column"""
        frame = ctk.CTkFrame(self, fg_color="transparent")
        frame.grid(row=1, column=col, padx=5, pady=(0, 15), sticky="n")

        # Label
        label = ctk.CTkLabel(
            frame,
            text="Toggle",
            font=(FONTS["family"], FONTS["size_small"]),
            text_color=COLORS["text_muted"]
        )
        label.pack(pady=(0, 5))

        # Key button
        self._toggle_button = self._create_key_button(
            frame,
            self.EMPTY_KEY,
            self._start_toggle_capture
        )
        self._toggle_button.pack()

    def _create_hold_column(self, col: int):
        """Create the Hold hotkey column"""
        frame = ctk.CTkFrame(self, fg_color="transparent")
        frame.grid(row=1, column=col, padx=5, pady=(0, 15), sticky="n")

        # Label
        label = ctk.CTkLabel(
            frame,
            text="Hold",
            font=(FONTS["family"], FONTS["size_small"]),
            text_color=COLORS["text_muted"]
        )
        label.pack(pady=(0, 5))

        # Key button
        self._hold_button = self._create_key_button(
            frame,
            self.EMPTY_KEY,
            self._start_hold_capture
        )
        self._hold_button.pack()

        # Enable checkbox
        checkbox = ctk.CTkCheckBox(
            frame,
            text="",
            variable=self._hold_enabled_var,
            command=self._on_hold_enabled_change,
            width=20,
            height=20,
            fg_color=COLORS["accent_orange"],
            hover_color=COLORS["accent_orange_hover"],
            border_color=COLORS["border"],
            checkmark_color=COLORS["bg_dark"],
            corner_radius=3
        )
        checkbox.pack(pady=(5, 0))

    def _create_focus_column(self, col: int):
        """Create the Focus hotkey column"""
        frame = ctk.CTkFrame(self, fg_color="transparent")
        frame.grid(row=1, column=col, padx=5, pady=(0, 15), sticky="n")

        # Label
        label = ctk.CTkLabel(
            frame,
            text="Focus",
            font=(FONTS["family"], FONTS["size_small"]),
            text_color=COLORS["text_muted"]
        )
        label.pack(pady=(0, 5))

        # Key button
        self._focus_button = self._create_key_button(
            frame,
            self.EMPTY_KEY,
            self._start_focus_capture
        )
        self._focus_button.pack()

        # Enable checkbox
        checkbox = ctk.CTkCheckBox(
            frame,
            text="",
            variable=self._focus_enabled_var,
            command=self._on_focus_enabled_change,
            width=20,
            height=20,
            fg_color=COLORS["accent_orange"],
            hover_color=COLORS["accent_orange_hover"],
            border_color=COLORS["border"],
            checkmark_color=COLORS["bg_dark"],
            corner_radius=3
        )
        checkbox.pack(pady=(5, 0))

    def _create_rclick_column(self, col: int):
        """Create the R-Hold (Right-click hold) hotkey column"""
        frame = ctk.CTkFrame(self, fg_color="transparent")
        frame.grid(row=1, column=col, padx=5, pady=(0, 15), sticky="n")

        # Label
        label = ctk.CTkLabel(
            frame,
            text="R-Hold",
            font=(FONTS["family"], FONTS["size_small"]),
            text_color=COLORS["text_muted"]
        )
        label.pack(pady=(0, 5))

        # Key button
        self._rclick_button = self._create_key_button(
            frame,
            self.EMPTY_KEY,
            self._start_rclick_capture
        )
        self._rclick_button.pack()

        # Enable checkbox
        checkbox = ctk.CTkCheckBox(
            frame,
            text="",
            variable=self._rclick_enabled_var,
            command=self._on_rclick_enabled_change,
            width=20,
            height=20,
            fg_color=COLORS["accent_orange"],
            hover_color=COLORS["accent_orange_hover"],
            border_color=COLORS["border"],
            checkmark_color=COLORS["bg_dark"],
            corner_radius=3
        )
        checkbox.pack(pady=(5, 0))

    # =========================================================================
    # Capture Start Methods
    # =========================================================================

    def _start_toggle_capture(self):
        """Start capturing toggle hotkey"""
        self._toggle_button.configure(text=self.CAPTURING_TEXT)
        if self._on_toggle_capture:
            self._on_toggle_capture()

    def _start_hold_capture(self):
        """Start capturing hold hotkey"""
        self._hold_button.configure(text=self.CAPTURING_TEXT)
        if self._on_hold_capture:
            self._on_hold_capture()

    def _start_focus_capture(self):
        """Start capturing focus hotkey"""
        self._focus_button.configure(text=self.CAPTURING_TEXT)
        if self._on_focus_capture:
            self._on_focus_capture()

    def _start_rclick_capture(self):
        """Start capturing r-hold hotkey"""
        self._rclick_button.configure(text=self.CAPTURING_TEXT)
        if self._on_rclick_capture:
            self._on_rclick_capture()

    # =========================================================================
    # Checkbox Change Handlers
    # =========================================================================

    def _on_hold_enabled_change(self):
        """Handle hold enabled checkbox change"""
        if self._on_hold_enabled:
            self._on_hold_enabled(self._hold_enabled_var.get())

    def _on_focus_enabled_change(self):
        """Handle focus enabled checkbox change"""
        if self._on_focus_enabled:
            self._on_focus_enabled(self._focus_enabled_var.get())

    def _on_rclick_enabled_change(self):
        """Handle r-hold enabled checkbox change"""
        if self._on_rclick_enabled:
            self._on_rclick_enabled(self._rclick_enabled_var.get())

    # =========================================================================
    # Public Methods - Set Key Display
    # =========================================================================

    def set_toggle_key(self, name: Optional[str]):
        """Set the toggle key display"""
        self._toggle_button.configure(text=name or self.EMPTY_KEY)

    def set_hold_key(self, name: Optional[str]):
        """Set the hold key display"""
        self._hold_button.configure(text=name or self.EMPTY_KEY)

    def set_focus_key(self, name: Optional[str]):
        """Set the focus key display"""
        self._focus_button.configure(text=name or self.EMPTY_KEY)

    def set_rclick_key(self, name: Optional[str]):
        """Set the r-hold key display"""
        self._rclick_button.configure(text=name or self.EMPTY_KEY)

    # =========================================================================
    # Public Methods - Get/Set Enabled States
    # =========================================================================

    def get_hold_enabled(self) -> bool:
        """Get hold enabled state"""
        return self._hold_enabled_var.get()

    def set_hold_enabled(self, enabled: bool):
        """Set hold enabled state"""
        self._hold_enabled_var.set(enabled)

    def get_focus_enabled(self) -> bool:
        """Get focus enabled state"""
        return self._focus_enabled_var.get()

    def set_focus_enabled(self, enabled: bool):
        """Set focus enabled state"""
        self._focus_enabled_var.set(enabled)

    def get_rclick_enabled(self) -> bool:
        """Get r-hold enabled state"""
        return self._rclick_enabled_var.get()

    def set_rclick_enabled(self, enabled: bool):
        """Set r-hold enabled state"""
        self._rclick_enabled_var.set(enabled)

    # =========================================================================
    # Configuration Methods
    # =========================================================================

    def set_config(self, config: dict):
        """
        Set full configuration from dictionary.

        Expected format:
        {
            "toggle": {"key": "F6"},
            "hold": {"key": "F7", "enabled": True},
            "focus_hold": {"key": None, "enabled": False},
            "right_click_hold": {"key": None, "enabled": False}
        }
        """
        if "toggle" in config:
            key_data = config["toggle"].get("key")
            key_name = key_data.get("name") if isinstance(key_data, dict) else key_data
            self.set_toggle_key(key_name)

        if "hold" in config:
            key_data = config["hold"].get("key")
            key_name = key_data.get("name") if isinstance(key_data, dict) else key_data
            self.set_hold_key(key_name)
            self.set_hold_enabled(config["hold"].get("enabled", False))

        if "focus_hold" in config:
            key_data = config["focus_hold"].get("key")
            key_name = key_data.get("name") if isinstance(key_data, dict) else key_data
            self.set_focus_key(key_name)
            self.set_focus_enabled(config["focus_hold"].get("enabled", False))

        if "right_click_hold" in config:
            key_data = config["right_click_hold"].get("key")
            key_name = key_data.get("name") if isinstance(key_data, dict) else key_data
            self.set_rclick_key(key_name)
            self.set_rclick_enabled(config["right_click_hold"].get("enabled", False))

    def get_config(self) -> dict:
        """Get current configuration as dictionary"""
        return {
            "toggle": {
                "key": self._toggle_button.cget("text") if self._toggle_button.cget("text") != self.EMPTY_KEY else None,
            },
            "hold": {
                "key": self._hold_button.cget("text") if self._hold_button.cget("text") != self.EMPTY_KEY else None,
                "enabled": self.get_hold_enabled(),
            },
            "focus_hold": {
                "key": self._focus_button.cget("text") if self._focus_button.cget("text") != self.EMPTY_KEY else None,
                "enabled": self.get_focus_enabled(),
            },
            "right_click_hold": {
                "key": self._rclick_button.cget("text") if self._rclick_button.cget("text") != self.EMPTY_KEY else None,
                "enabled": self.get_rclick_enabled(),
            },
        }

    def cancel_capture(self, target: str):
        """
        Cancel capture mode and restore button text.

        Args:
            target: Which hotkey was being captured ("toggle", "hold", "focus_hold", "right_click_hold")
        """
        if target == "toggle":
            current = self._toggle_button.cget("text")
            if current == self.CAPTURING_TEXT:
                self._toggle_button.configure(text=self.EMPTY_KEY)
        elif target == "hold":
            current = self._hold_button.cget("text")
            if current == self.CAPTURING_TEXT:
                self._hold_button.configure(text=self.EMPTY_KEY)
        elif target == "focus_hold":
            current = self._focus_button.cget("text")
            if current == self.CAPTURING_TEXT:
                self._focus_button.configure(text=self.EMPTY_KEY)
        elif target == "right_click_hold":
            current = self._rclick_button.cget("text")
            if current == self.CAPTURING_TEXT:
                self._rclick_button.configure(text=self.EMPTY_KEY)
