"""
Main Application Window - Two-panel layout with scrollable frames

SCARMONIT themed key spammer application with advanced hotkey capture system.

Layout:
- Title (top, spanning both columns)
- Left Panel: Hotkeys, Key Config, Quick Actions (scrollable)
- Right Panel: Profiles, Focus/Timing, Controls (scrollable)
"""

import customtkinter as ctk
from typing import Optional
import sys
import os

# Add parent directory to path for imports
sys.path.insert(0, os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

from gui.theme import COLORS, FONTS, WIDGET_CONFIG
from gui.components.left_panel import LeftPanel
from gui.components.right_panel import RightPanel

from core.spammer import Spammer, SpammerState, SpammerConfig, get_spammer
from core.hotkey_capture_manager import HotkeyCaptureManager, get_hotkey_capture_manager
from core.captured_key import CapturedKey
from utils.config_manager import ConfigManager, ProfileData, HotkeysConfig, get_config_manager


class SpammerApp(ctk.CTk):
    """
    Main application window for the SCARMONIT Key Spammer.

    Features two-panel layout with scrollable frames for better organization
    and visibility of all configuration options.
    """

    def __init__(self):
        super().__init__()

        # Window configuration - wider for two panels
        self.title("SCARMONIT SPAMMER")
        self.geometry("750x550")
        self.minsize(700, 450)
        self.configure(fg_color=COLORS["bg_dark"])

        # Set appearance mode
        ctk.set_appearance_mode("dark")

        # Initialize core components
        self.spammer = get_spammer()
        self.hotkey_manager = get_hotkey_capture_manager()
        self.config_manager = get_config_manager()

        # Current profile
        self.current_profile: Optional[ProfileData] = None

        # Track if spammer was running before hold
        self._was_running_before_hold = False

        # Create GUI components
        self._create_widgets()

        # Setup callbacks
        self._setup_callbacks()

        # Load initial profile
        self._load_initial_profile()

        # Start hotkey listener
        self.hotkey_manager.start()

        # Handle window close
        self.protocol("WM_DELETE_WINDOW", self._on_close)

    def _create_widgets(self):
        """Create and layout all GUI components in two-panel layout"""
        # Configure main grid - 2 columns for panels
        self.grid_columnconfigure(0, weight=1)
        self.grid_columnconfigure(1, weight=1)
        self.grid_rowconfigure(0, weight=0)  # Title
        self.grid_rowconfigure(1, weight=1)  # Panels (expandable)

        # Title/header
        title_label = ctk.CTkLabel(
            self,
            text="SCARMONIT SPAMMER",
            font=(FONTS["family"], FONTS["size_title"], FONTS["weight_bold"]),
            text_color=COLORS["accent_orange"]
        )
        title_label.grid(row=0, column=0, columnspan=2, pady=(15, 10))

        # Left Panel (Key Configuration)
        self.left_panel = LeftPanel(
            self,
            # Hotkey callbacks
            on_toggle_capture_start=self._on_toggle_capture_start,
            on_hold_capture_start=self._on_hold_capture_start,
            on_focus_capture_start=self._on_focus_capture_start,
            on_rclick_capture_start=self._on_rclick_capture_start,
            on_hold_enabled_change=self._on_hold_enabled_change,
            on_focus_enabled_change=self._on_focus_enabled_change,
            on_rclick_enabled_change=self._on_rclick_enabled_change,
            # Key config callbacks
            on_keys_change=self._on_keys_change,
            # Quick actions callbacks
            on_quick_actions_change=self._on_quick_actions_change
        )
        self.left_panel.grid(row=1, column=0, sticky="nsew", padx=(15, 7), pady=(0, 15))

        # Right Panel (Settings & Controls)
        self.right_panel = RightPanel(
            self,
            profiles=self.config_manager.list_profiles(),
            on_profile_select=self._on_profile_select,
            on_profile_save=self._on_profile_save,
            on_profile_delete=self._on_profile_delete,
            on_focus_timing_change=self._on_focus_timing_change,
            on_start=self._on_start,
            on_stop=self._on_stop
        )
        self.right_panel.grid(row=1, column=1, sticky="nsew", padx=(7, 15), pady=(0, 15))

        # Initial button states
        self.right_panel.set_stopped()

    def _setup_callbacks(self):
        """Setup callbacks between components"""
        # Spammer state change callback
        self.spammer.on_state_change(self._on_spammer_state_change)

        # Hotkey capture manager callbacks
        self.hotkey_manager.on_toggle(self._on_toggle_hotkey)
        self.hotkey_manager.on_hold_start(self._on_hold_start)
        self.hotkey_manager.on_hold_end(self._on_hold_end)
        self.hotkey_manager.on_focus_hold_start(self._on_focus_hold_start)
        self.hotkey_manager.on_focus_hold_end(self._on_focus_hold_end)
        self.hotkey_manager.on_right_click_hold_start(self._on_rclick_hold_start)
        self.hotkey_manager.on_right_click_hold_end(self._on_rclick_hold_end)
        self.hotkey_manager.on_key_captured(self._on_key_captured)

    def _load_initial_profile(self):
        """Load the initial/default profile"""
        profiles = self.config_manager.list_profiles()
        if profiles:
            self._load_profile(profiles[0])

    def _load_profile(self, name: str):
        """Load a profile and update GUI"""
        profile = self.config_manager.load_profile(name)
        if profile:
            self.current_profile = profile
            self._apply_profile_to_gui(profile)
            self._apply_profile_to_spammer(profile)

    def _apply_profile_to_gui(self, profile: ProfileData):
        """Apply profile settings to GUI components"""
        # Left panel settings
        self.left_panel.set_keys(profile.keys)
        self.left_panel.set_quick_actions(
            spacebar=profile.spacebar_enabled,
            left_click=profile.left_click_enabled,
            right_click=profile.right_click_enabled
        )

        # Apply hotkey configuration
        hotkeys_config = profile.hotkeys.to_dict()
        self.left_panel.set_hotkeys_config(hotkeys_config)
        self._apply_hotkeys_to_manager(profile)

        # Right panel settings
        self.right_panel.set_target_windows(profile.target_windows)
        self.right_panel.set_pause_on_unfocus(profile.pause_on_unfocus)
        self.right_panel.set_current_profile(profile.name)

    def _apply_profile_to_spammer(self, profile: ProfileData):
        """Apply profile settings to spammer"""
        config = SpammerConfig(
            keys=profile.keys,
            spacebar_enabled=profile.spacebar_enabled,
            left_click_enabled=profile.left_click_enabled,
            right_click_enabled=profile.right_click_enabled,
            target_windows=profile.target_windows,
            pause_on_unfocus=profile.pause_on_unfocus
        )
        self.spammer.set_config(config)

    def _get_current_settings(self) -> ProfileData:
        """Get current settings from GUI as ProfileData"""
        spacebar, left_click, right_click = self.left_panel.get_quick_actions()
        target_windows, pause_on_unfocus = self.right_panel.get_focus_settings()

        # Get hotkeys config from UI
        hotkeys_dict = self.left_panel.get_hotkeys_config()
        hotkeys = HotkeysConfig.from_dict(hotkeys_dict)

        return ProfileData(
            name=self.right_panel.get_current_profile(),
            keys=self.left_panel.get_keys(),
            spacebar_enabled=spacebar,
            left_click_enabled=left_click,
            right_click_enabled=right_click,
            target_windows=target_windows,
            pause_on_unfocus=pause_on_unfocus,
            hotkeys=hotkeys
        )

    def _update_spammer_from_gui(self):
        """Update spammer settings from current GUI state"""
        profile = self._get_current_settings()
        self._apply_profile_to_spammer(profile)

    # =========================================================================
    # Profile Event Handlers
    # =========================================================================

    def _on_profile_select(self, name: str):
        """Handle profile selection"""
        self._load_profile(name)

    def _on_profile_save(self, name: str):
        """Handle profile save"""
        profile = self._get_current_settings()
        profile.name = name
        self.config_manager.save_profile(profile)
        self.right_panel.set_profiles(self.config_manager.list_profiles())
        self.right_panel.set_current_profile(name)

    def _on_profile_delete(self, name: str):
        """Handle profile delete"""
        profiles = self.config_manager.list_profiles()
        if len(profiles) > 1:
            self.config_manager.delete_profile(name)
            self.right_panel.set_profiles(self.config_manager.list_profiles())
            # Load first remaining profile
            remaining = self.config_manager.list_profiles()
            if remaining:
                self._load_profile(remaining[0])

    # =========================================================================
    # Settings Change Handlers
    # =========================================================================

    def _on_keys_change(self, keys: list):
        """Handle keys list change"""
        self._update_spammer_from_gui()

    def _on_quick_actions_change(self, spacebar: bool, left_click: bool, right_click: bool):
        """Handle quick actions change"""
        self._update_spammer_from_gui()

    def _on_focus_timing_change(self, target_windows: str, pause_on_unfocus: bool):
        """Handle focus timing settings change"""
        self._update_spammer_from_gui()

    # =========================================================================
    # Control Handlers
    # =========================================================================

    def _on_start(self):
        """Handle start button/hotkey"""
        if self.spammer.state == SpammerState.STOPPED:
            self._update_spammer_from_gui()
            self.spammer.start()

    def _on_stop(self):
        """Handle stop button/hotkey"""
        if self.spammer.state != SpammerState.STOPPED:
            self.spammer.stop()

    def _on_spammer_state_change(self, state: SpammerState):
        """Handle spammer state changes"""
        # Update GUI on main thread
        self.after(0, lambda: self._update_status_display(state))

    def _update_status_display(self, state: SpammerState):
        """Update the status display based on spammer state"""
        if state == SpammerState.RUNNING:
            self.right_panel.set_running()
        elif state == SpammerState.STOPPED:
            self.right_panel.set_stopped()
        elif state == SpammerState.PAUSED:
            self.right_panel.set_paused()

    # =========================================================================
    # Hotkey System Methods
    # =========================================================================

    def _apply_hotkeys_to_manager(self, profile: ProfileData):
        """Apply profile hotkey settings to the hotkey capture manager"""
        hotkeys = profile.hotkeys

        # Toggle hotkey
        if hotkeys.toggle.key_name:
            toggle_key = CapturedKey.from_name(hotkeys.toggle.key_name)
            self.hotkey_manager.set_hotkey(HotkeyCaptureManager.TOGGLE, toggle_key)

        # Hold hotkey
        if hotkeys.hold.key_name:
            hold_key = CapturedKey.from_name(hotkeys.hold.key_name)
            self.hotkey_manager.set_hotkey(HotkeyCaptureManager.HOLD, hold_key)
        self.hotkey_manager.set_enabled(HotkeyCaptureManager.HOLD, hotkeys.hold.enabled)

        # Focus hold hotkey
        if hotkeys.focus_hold.key_name:
            focus_key = CapturedKey.from_name(hotkeys.focus_hold.key_name)
            self.hotkey_manager.set_hotkey(HotkeyCaptureManager.FOCUS_HOLD, focus_key)
        self.hotkey_manager.set_enabled(HotkeyCaptureManager.FOCUS_HOLD, hotkeys.focus_hold.enabled)

        # Right-click hold hotkey
        if hotkeys.right_click_hold.key_name:
            rclick_key = CapturedKey.from_name(hotkeys.right_click_hold.key_name)
            self.hotkey_manager.set_hotkey(HotkeyCaptureManager.RIGHT_CLICK_HOLD, rclick_key)
        self.hotkey_manager.set_enabled(HotkeyCaptureManager.RIGHT_CLICK_HOLD, hotkeys.right_click_hold.enabled)

    # =========================================================================
    # Capture Start Callbacks (from UI)
    # =========================================================================

    def _on_toggle_capture_start(self):
        """Start capture mode for toggle hotkey"""
        self.hotkey_manager.start_capture(HotkeyCaptureManager.TOGGLE)

    def _on_hold_capture_start(self):
        """Start capture mode for hold hotkey"""
        self.hotkey_manager.start_capture(HotkeyCaptureManager.HOLD)

    def _on_focus_capture_start(self):
        """Start capture mode for focus hold hotkey"""
        self.hotkey_manager.start_capture(HotkeyCaptureManager.FOCUS_HOLD)

    def _on_rclick_capture_start(self):
        """Start capture mode for right-click hold hotkey"""
        self.hotkey_manager.start_capture(HotkeyCaptureManager.RIGHT_CLICK_HOLD)

    # =========================================================================
    # Enable/Disable Callbacks (from UI checkboxes)
    # =========================================================================

    def _on_hold_enabled_change(self, enabled: bool):
        """Handle hold enabled checkbox change"""
        self.hotkey_manager.set_enabled(HotkeyCaptureManager.HOLD, enabled)

    def _on_focus_enabled_change(self, enabled: bool):
        """Handle focus enabled checkbox change"""
        self.hotkey_manager.set_enabled(HotkeyCaptureManager.FOCUS_HOLD, enabled)

    def _on_rclick_enabled_change(self, enabled: bool):
        """Handle r-hold enabled checkbox change"""
        self.hotkey_manager.set_enabled(HotkeyCaptureManager.RIGHT_CLICK_HOLD, enabled)

    # =========================================================================
    # Key Captured Callback
    # =========================================================================

    def _on_key_captured(self, target: str, captured_key: CapturedKey):
        """Handle when a key is captured during capture mode"""
        # Update UI on main thread
        def update_ui():
            if target == HotkeyCaptureManager.TOGGLE:
                self.left_panel.set_toggle_key(captured_key.name)
            elif target == HotkeyCaptureManager.HOLD:
                self.left_panel.set_hold_key(captured_key.name)
            elif target == HotkeyCaptureManager.FOCUS_HOLD:
                self.left_panel.set_focus_key(captured_key.name)
            elif target == HotkeyCaptureManager.RIGHT_CLICK_HOLD:
                self.left_panel.set_rclick_key(captured_key.name)

        self.after(0, update_ui)

    # =========================================================================
    # Hotkey Action Callbacks
    # =========================================================================

    def _on_toggle_hotkey(self):
        """Handle toggle hotkey pressed - toggle spammer on/off"""
        if self.spammer.state == SpammerState.STOPPED:
            self._update_spammer_from_gui()
            self.spammer.start()
        else:
            self.spammer.stop()

    def _on_hold_start(self):
        """Handle hold hotkey pressed - start spamming"""
        self._was_running_before_hold = self.spammer.state == SpammerState.RUNNING
        if not self._was_running_before_hold:
            self._update_spammer_from_gui()
            self.spammer.start()

    def _on_hold_end(self):
        """Handle hold hotkey released - stop spamming"""
        if not self._was_running_before_hold:
            self.spammer.stop()

    def _on_focus_hold_start(self, focus_key: CapturedKey):
        """Handle focus hold hotkey pressed - spam only the focus key"""
        # Save current state and start spamming only the focus key
        self._was_running_before_hold = self.spammer.state == SpammerState.RUNNING
        if not self._was_running_before_hold:
            target_windows, pause_on_unfocus = self.right_panel.get_focus_settings()
            # Create config with only the focus key
            config = SpammerConfig(
                keys=[focus_key.name.lower()] if not focus_key.is_mouse else [],
                spacebar_enabled=False,
                left_click_enabled=focus_key.is_mouse and focus_key.name == "MOUSE_LEFT",
                right_click_enabled=focus_key.is_mouse and focus_key.name == "MOUSE_RIGHT",
                target_windows=target_windows,
                pause_on_unfocus=pause_on_unfocus
            )
            self.spammer.set_config(config)
            self.spammer.start()

    def _on_focus_hold_end(self):
        """Handle focus hold hotkey released"""
        if not self._was_running_before_hold:
            self.spammer.stop()
            # Restore original config
            self._update_spammer_from_gui()

    def _on_rclick_hold_start(self):
        """Handle right-click hold hotkey pressed - spam only right-click"""
        self._was_running_before_hold = self.spammer.state == SpammerState.RUNNING
        if not self._was_running_before_hold:
            target_windows, pause_on_unfocus = self.right_panel.get_focus_settings()
            # Create config with only right-click
            config = SpammerConfig(
                keys=[],
                spacebar_enabled=False,
                left_click_enabled=False,
                right_click_enabled=True,
                target_windows=target_windows,
                pause_on_unfocus=pause_on_unfocus
            )
            self.spammer.set_config(config)
            self.spammer.start()

    def _on_rclick_hold_end(self):
        """Handle right-click hold hotkey released"""
        if not self._was_running_before_hold:
            self.spammer.stop()
            # Restore original config
            self._update_spammer_from_gui()

    # =========================================================================
    # Window Close
    # =========================================================================

    def _on_close(self):
        """Handle window close"""
        # Stop spammer if running
        self.spammer.stop()

        # Stop hotkey listener
        self.hotkey_manager.stop()

        # Destroy window
        self.destroy()


def run_app():
    """Entry point to run the application"""
    app = SpammerApp()
    app.mainloop()


if __name__ == "__main__":
    run_app()
