"""
Window Focus Module - Detects target windows and monitors focus state

Uses Win32 API (pywin32) on Windows to:
- Find windows by partial title match
- Check if target windows are currently focused
- Support comma-separated list of target window names
"""

import sys
from typing import List, Optional, Tuple


class WindowFocusManager:
    """
    Manages window focus detection for the "Pause when unfocused" feature.

    Supports partial window title matching and multiple target windows.
    """

    def __init__(self):
        """Initialize the window focus manager"""
        self.is_windows = sys.platform == 'win32'
        self.target_windows: List[str] = []
        self._win32gui = None

        if self.is_windows:
            self._init_win32()

    def _init_win32(self):
        """Initialize Win32 API access"""
        try:
            import win32gui
            self._win32gui = win32gui
            print("[WindowFocus] Win32 API initialized successfully")
        except ImportError:
            print("[WindowFocus] WARNING: pywin32 not installed. Window focus detection disabled.")
            self._win32gui = None

    def set_target_windows(self, windows: str):
        """
        Set target windows from comma-separated string.

        Args:
            windows: Comma-separated list of window names (e.g., "Diablo IV, Path of Exile")
        """
        if not windows or not windows.strip():
            self.target_windows = []
            return

        # Parse comma-separated list, strip whitespace, filter empty
        self.target_windows = [
            w.strip().lower()
            for w in windows.split(',')
            if w.strip()
        ]

    def get_foreground_window_title(self) -> str:
        """
        Get the title of the currently focused window.

        Returns:
            str: Window title, or empty string if not available
        """
        if not self.is_windows or not self._win32gui:
            return ""

        try:
            hwnd = self._win32gui.GetForegroundWindow()
            return self._win32gui.GetWindowText(hwnd)
        except Exception:
            return ""

    def is_target_window_focused(self) -> bool:
        """
        Check if any of the target windows is currently focused.

        Returns:
            bool: True if a target window is focused (or no targets set), False otherwise
        """
        # If no targets specified, always return True (don't pause)
        if not self.target_windows:
            return True

        # If not Windows or win32gui not available, assume focused
        if not self.is_windows or not self._win32gui:
            return True

        current_title = self.get_foreground_window_title().lower()

        # Check if current window title contains any target window name
        for target in self.target_windows:
            if target in current_title:
                return True

        return False

    def find_window_by_title(self, partial_title: str) -> Optional[int]:
        """
        Find a window handle by partial title match.

        Args:
            partial_title: Partial window title to search for

        Returns:
            int: Window handle (hwnd) if found, None otherwise
        """
        if not self.is_windows or not self._win32gui:
            return None

        partial_lower = partial_title.lower()
        found_hwnd = None

        def enum_callback(hwnd, _):
            nonlocal found_hwnd
            if self._win32gui.IsWindowVisible(hwnd):
                title = self._win32gui.GetWindowText(hwnd).lower()
                if partial_lower in title:
                    found_hwnd = hwnd
                    return False  # Stop enumeration
            return True

        try:
            self._win32gui.EnumWindows(enum_callback, None)
        except Exception:
            pass

        return found_hwnd

    def get_all_visible_windows(self) -> List[Tuple[int, str]]:
        """
        Get a list of all visible windows with their titles.

        Returns:
            List of (hwnd, title) tuples
        """
        if not self.is_windows or not self._win32gui:
            return []

        windows = []

        def enum_callback(hwnd, _):
            if self._win32gui.IsWindowVisible(hwnd):
                title = self._win32gui.GetWindowText(hwnd)
                if title:  # Only include windows with titles
                    windows.append((hwnd, title))
            return True

        try:
            self._win32gui.EnumWindows(enum_callback, None)
        except Exception:
            pass

        return windows

    def is_available(self) -> bool:
        """Check if window focus detection is available"""
        return self.is_windows and self._win32gui is not None

    def get_status(self) -> dict:
        """Get current status information"""
        return {
            'available': self.is_available(),
            'platform': sys.platform,
            'target_windows': self.target_windows,
            'current_foreground': self.get_foreground_window_title(),
            'is_target_focused': self.is_target_window_focused(),
        }


# Singleton instance
_default_manager = None


def get_window_focus_manager() -> WindowFocusManager:
    """Get the default window focus manager instance"""
    global _default_manager
    if _default_manager is None:
        _default_manager = WindowFocusManager()
    return _default_manager


def is_target_focused() -> bool:
    """Convenience function to check if target window is focused"""
    return get_window_focus_manager().is_target_window_focused()


def set_targets(windows: str):
    """Convenience function to set target windows"""
    get_window_focus_manager().set_target_windows(windows)
