"""
Spammer Module - Main automation thread for key/mouse spamming

Coordinates the humanizer, input handler, and window focus detection
to perform human-like automated inputs.
"""

import random
import threading
import time
from dataclasses import dataclass, field
from typing import Callable, List, Optional
from enum import Enum, auto

from .humanizer import Humanizer, get_humanizer
from .input_handler import InputHandler, get_input_handler
from .window_focus import WindowFocusManager, get_window_focus_manager


class SpammerState(Enum):
    """Current state of the spammer"""
    STOPPED = auto()
    RUNNING = auto()
    PAUSED = auto()  # Paused due to window focus


@dataclass
class SpammerConfig:
    """Configuration for the spammer"""
    # Keys to spam (list of key names)
    keys: List[str] = field(default_factory=list)

    # Quick actions
    spacebar_enabled: bool = False
    left_click_enabled: bool = False
    right_click_enabled: bool = False

    # Window focus settings
    target_windows: str = ""  # Comma-separated window names
    pause_on_unfocus: bool = True


class Spammer:
    """
    Main spammer class that coordinates all automation.

    Runs in a separate thread to avoid blocking the GUI.
    Supports pause/resume, window focus detection, and human-like timing.
    """

    def __init__(
        self,
        humanizer: Optional[Humanizer] = None,
        input_handler: Optional[InputHandler] = None,
        window_focus: Optional[WindowFocusManager] = None
    ):
        """Initialize the spammer with optional custom components"""
        self.humanizer = humanizer or get_humanizer()
        self.input_handler = input_handler or get_input_handler()
        self.window_focus = window_focus or get_window_focus_manager()

        self.config = SpammerConfig()
        self.state = SpammerState.STOPPED

        # Threading
        self._thread: Optional[threading.Thread] = None
        self._stop_event = threading.Event()
        self._pause_event = threading.Event()
        self._pause_event.set()  # Not paused by default

        # Callbacks
        self._on_state_change: Optional[Callable[[SpammerState], None]] = None
        self._on_action: Optional[Callable[[str], None]] = None

        # Statistics
        self.action_count = 0
        self.session_start: Optional[float] = None

    def set_config(self, config: SpammerConfig):
        """Update the spammer configuration"""
        self.config = config
        self.window_focus.set_target_windows(config.target_windows)

    def set_keys(self, keys: List[str]):
        """Set the keys to spam"""
        self.config.keys = keys

    def set_quick_actions(
        self,
        spacebar: bool = False,
        left_click: bool = False,
        right_click: bool = False
    ):
        """Set quick action toggles"""
        self.config.spacebar_enabled = spacebar
        self.config.left_click_enabled = left_click
        self.config.right_click_enabled = right_click

    def set_window_settings(self, target_windows: str, pause_on_unfocus: bool):
        """Configure window focus settings"""
        self.config.target_windows = target_windows
        self.config.pause_on_unfocus = pause_on_unfocus
        self.window_focus.set_target_windows(target_windows)

    def on_state_change(self, callback: Callable[[SpammerState], None]):
        """Register callback for state changes"""
        self._on_state_change = callback

    def on_action(self, callback: Callable[[str], None]):
        """Register callback for each action performed"""
        self._on_action = callback

    def _set_state(self, state: SpammerState):
        """Update state and notify callback"""
        self.state = state
        if self._on_state_change:
            try:
                self._on_state_change(state)
            except Exception as e:
                print(f"[Spammer] Error in state change callback: {e}")

    def _get_enabled_actions(self) -> List[str]:
        """Get list of all enabled actions (keys + quick actions)"""
        actions = list(self.config.keys)

        if self.config.spacebar_enabled:
            actions.append('space')
        if self.config.left_click_enabled:
            actions.append('left_click')
        if self.config.right_click_enabled:
            actions.append('right_click')

        return actions

    def _perform_action(self, action: str):
        """Perform a single action with human-like timing"""
        duration = self.humanizer.get_press_duration()

        # Use perform_action which handles both keyboard and mouse button strings
        self.input_handler.perform_action(action, duration)

        self.action_count += 1

        if self._on_action:
            try:
                self._on_action(action)
            except Exception:
                pass

    def _run_loop(self):
        """Main spammer loop (runs in thread)"""
        self.action_count = 0
        self.session_start = time.time()
        self.humanizer.reset()

        print("[Spammer] Started spamming")

        while not self._stop_event.is_set():
            # Check pause event
            self._pause_event.wait()

            if self._stop_event.is_set():
                break

            # Check window focus if enabled
            if self.config.pause_on_unfocus:
                if not self.window_focus.is_target_window_focused():
                    if self.state != SpammerState.PAUSED:
                        self._set_state(SpammerState.PAUSED)
                    time.sleep(0.1)  # Check focus every 100ms
                    continue
                elif self.state == SpammerState.PAUSED:
                    self._set_state(SpammerState.RUNNING)

            # Get enabled actions
            actions = self._get_enabled_actions()
            if not actions:
                time.sleep(0.1)
                continue

            # Shuffle for unpredictability
            random.shuffle(actions)

            # Perform each action
            for action in actions:
                if self._stop_event.is_set():
                    break

                self._perform_action(action)

                # Human-like delay between actions
                delay = self.humanizer.get_delay()
                time.sleep(delay)

        print(f"[Spammer] Stopped. Total actions: {self.action_count}")

    def start(self):
        """Start spamming"""
        if self.state == SpammerState.RUNNING:
            return

        self._stop_event.clear()
        self._pause_event.set()

        self._thread = threading.Thread(target=self._run_loop, daemon=True)
        self._thread.start()

        self._set_state(SpammerState.RUNNING)

    def stop(self):
        """Stop spamming"""
        if self.state == SpammerState.STOPPED:
            return

        self._stop_event.set()
        self._pause_event.set()  # Unblock if paused

        if self._thread:
            self._thread.join(timeout=1.0)
            self._thread = None

        self._set_state(SpammerState.STOPPED)

    def pause(self):
        """Pause spamming"""
        self._pause_event.clear()
        if self.state == SpammerState.RUNNING:
            self._set_state(SpammerState.PAUSED)

    def resume(self):
        """Resume spamming"""
        self._pause_event.set()
        if self.state == SpammerState.PAUSED:
            self._set_state(SpammerState.RUNNING)

    def toggle(self):
        """Toggle between running and stopped"""
        if self.state == SpammerState.STOPPED:
            self.start()
        else:
            self.stop()

    def is_running(self) -> bool:
        """Check if spammer is currently running"""
        return self.state == SpammerState.RUNNING

    def is_paused(self) -> bool:
        """Check if spammer is currently paused"""
        return self.state == SpammerState.PAUSED

    def get_stats(self) -> dict:
        """Get current spammer statistics"""
        session_duration = 0
        if self.session_start:
            session_duration = time.time() - self.session_start

        return {
            'state': self.state.name,
            'action_count': self.action_count,
            'session_duration': session_duration,
            'enabled_actions': self._get_enabled_actions(),
            'humanizer_stats': self.humanizer.get_stats(),
        }


# Singleton instance
_default_spammer = None


def get_spammer() -> Spammer:
    """Get the default spammer instance"""
    global _default_spammer
    if _default_spammer is None:
        _default_spammer = Spammer()
    return _default_spammer
