"""
Input Handler Module - Platform abstraction for keyboard and mouse automation

Uses pydirectinput on Windows for maximum speed (SendInput API),
falls back to pynput for cross-platform compatibility.
"""

import sys
import time
from typing import Optional, Union
from enum import Enum, auto

from .humanizer import get_humanizer, Humanizer


class MouseButton(Enum):
    """Mouse button enum for cross-platform compatibility"""
    LEFT = auto()
    RIGHT = auto()
    MIDDLE = auto()


class InputHandler:
    """
    Unified interface for keyboard and mouse input automation.

    Automatically selects the fastest available backend:
    - Windows: pydirectinput (uses SendInput API, works with DirectX games)
    - Other: pynput (cross-platform)
    """

    # Special key mappings for consistent naming across platforms
    SPECIAL_KEYS = {
        'space': 'space',
        'spacebar': 'space',
        'enter': 'enter',
        'return': 'enter',
        'tab': 'tab',
        'escape': 'esc',
        'esc': 'esc',
        'shift': 'shift',
        'ctrl': 'ctrl',
        'control': 'ctrl',
        'alt': 'alt',
        'backspace': 'backspace',
        'delete': 'delete',
        'up': 'up',
        'down': 'down',
        'left': 'left',
        'right': 'right',
        'f1': 'f1', 'f2': 'f2', 'f3': 'f3', 'f4': 'f4',
        'f5': 'f5', 'f6': 'f6', 'f7': 'f7', 'f8': 'f8',
        'f9': 'f9', 'f10': 'f10', 'f11': 'f11', 'f12': 'f12',
    }

    def __init__(self, humanizer: Optional[Humanizer] = None):
        """
        Initialize the input handler.

        Args:
            humanizer: Optional custom humanizer instance. Uses default if None.
        """
        self.humanizer = humanizer or get_humanizer()
        self.is_windows = sys.platform == 'win32'

        # Initialize the appropriate backend
        self._init_backend()

    def _init_backend(self):
        """Initialize the platform-specific input backend"""
        if self.is_windows:
            try:
                import pydirectinput
                # Disable artificial pause for maximum speed
                pydirectinput.PAUSE = None
                self._keyboard = pydirectinput
                self._mouse = pydirectinput
                self._backend = 'pydirectinput'
                print("[InputHandler] Using pydirectinput (Windows - fastest)")
            except ImportError:
                self._init_pynput_backend()
        else:
            self._init_pynput_backend()

    def _init_pynput_backend(self):
        """Initialize pynput as fallback backend"""
        from pynput.keyboard import Controller as KeyboardController, Key
        from pynput.mouse import Controller as MouseController, Button

        self._keyboard_controller = KeyboardController()
        self._mouse_controller = MouseController()
        self._pynput_key = Key
        self._pynput_button = Button
        self._backend = 'pynput'
        print("[InputHandler] Using pynput (cross-platform)")

    def _normalize_key(self, key: str) -> str:
        """Normalize key name to consistent format"""
        key_lower = key.lower().strip()
        return self.SPECIAL_KEYS.get(key_lower, key_lower)

    def _get_pynput_key(self, key: str):
        """Convert key string to pynput Key enum or character"""
        key = self._normalize_key(key)

        # Check if it's a special key
        pynput_special_keys = {
            'space': self._pynput_key.space,
            'enter': self._pynput_key.enter,
            'tab': self._pynput_key.tab,
            'esc': self._pynput_key.esc,
            'shift': self._pynput_key.shift,
            'ctrl': self._pynput_key.ctrl,
            'alt': self._pynput_key.alt,
            'backspace': self._pynput_key.backspace,
            'delete': self._pynput_key.delete,
            'up': self._pynput_key.up,
            'down': self._pynput_key.down,
            'left': self._pynput_key.left,
            'right': self._pynput_key.right,
            'f1': self._pynput_key.f1, 'f2': self._pynput_key.f2,
            'f3': self._pynput_key.f3, 'f4': self._pynput_key.f4,
            'f5': self._pynput_key.f5, 'f6': self._pynput_key.f6,
            'f7': self._pynput_key.f7, 'f8': self._pynput_key.f8,
            'f9': self._pynput_key.f9, 'f10': self._pynput_key.f10,
            'f11': self._pynput_key.f11, 'f12': self._pynput_key.f12,
        }

        if key in pynput_special_keys:
            return pynput_special_keys[key]

        # Regular character key
        return key

    def press_key(self, key: str, duration: Optional[float] = None):
        """
        Press and release a key with human-like timing.

        Args:
            key: Key to press (e.g., 'a', 'space', 'enter')
            duration: Optional custom press duration. Uses humanizer if None.
        """
        key = self._normalize_key(key)
        hold_time = duration if duration is not None else self.humanizer.get_press_duration()

        if self._backend == 'pydirectinput':
            self._keyboard.keyDown(key, _pause=False)
            time.sleep(hold_time)
            self._keyboard.keyUp(key, _pause=False)
        else:
            pynput_key = self._get_pynput_key(key)
            self._keyboard_controller.press(pynput_key)
            time.sleep(hold_time)
            self._keyboard_controller.release(pynput_key)

    def key_down(self, key: str):
        """Press a key down (without releasing)"""
        key = self._normalize_key(key)

        if self._backend == 'pydirectinput':
            self._keyboard.keyDown(key, _pause=False)
        else:
            pynput_key = self._get_pynput_key(key)
            self._keyboard_controller.press(pynput_key)

    def key_up(self, key: str):
        """Release a key"""
        key = self._normalize_key(key)

        if self._backend == 'pydirectinput':
            self._keyboard.keyUp(key, _pause=False)
        else:
            pynput_key = self._get_pynput_key(key)
            self._keyboard_controller.release(pynput_key)

    def click(self, button: MouseButton = MouseButton.LEFT, duration: Optional[float] = None):
        """
        Perform a mouse click with human-like timing.

        Args:
            button: Which mouse button to click
            duration: Optional custom click duration. Uses humanizer if None.
        """
        hold_time = duration if duration is not None else self.humanizer.get_press_duration()

        if self._backend == 'pydirectinput':
            button_map = {
                MouseButton.LEFT: ('left', self._mouse.mouseDown, self._mouse.mouseUp),
                MouseButton.RIGHT: ('right', self._mouse.mouseDown, self._mouse.mouseUp),
                MouseButton.MIDDLE: ('middle', self._mouse.mouseDown, self._mouse.mouseUp),
            }
            btn_name, down_func, up_func = button_map[button]
            down_func(button=btn_name, _pause=False)
            time.sleep(hold_time)
            up_func(button=btn_name, _pause=False)
        else:
            button_map = {
                MouseButton.LEFT: self._pynput_button.left,
                MouseButton.RIGHT: self._pynput_button.right,
                MouseButton.MIDDLE: self._pynput_button.middle,
            }
            pynput_button = button_map[button]
            self._mouse_controller.press(pynput_button)
            time.sleep(hold_time)
            self._mouse_controller.release(pynput_button)

    def left_click(self, duration: Optional[float] = None):
        """Perform a left mouse click"""
        self.click(MouseButton.LEFT, duration)

    def right_click(self, duration: Optional[float] = None):
        """Perform a right mouse click"""
        self.click(MouseButton.RIGHT, duration)

    def middle_click(self, duration: Optional[float] = None):
        """Perform a middle mouse click"""
        self.click(MouseButton.MIDDLE, duration)

    def perform_action(self, action: Union[str, MouseButton], duration: Optional[float] = None):
        """
        Perform an action (key press or mouse click) with human-like timing.

        Args:
            action: Key name (str) or MouseButton enum
            duration: Optional custom duration
        """
        if isinstance(action, MouseButton):
            self.click(action, duration)
            return

        action_lower = action.lower().strip()

        # Mouse button string mappings
        mouse_actions = {
            'leftclick': MouseButton.LEFT,
            'left_click': MouseButton.LEFT,
            'left click': MouseButton.LEFT,
            'mouse_left': MouseButton.LEFT,
            'rightclick': MouseButton.RIGHT,
            'right_click': MouseButton.RIGHT,
            'right click': MouseButton.RIGHT,
            'mouse_right': MouseButton.RIGHT,
            'middleclick': MouseButton.MIDDLE,
            'middle_click': MouseButton.MIDDLE,
            'middle click': MouseButton.MIDDLE,
            'mouse_middle': MouseButton.MIDDLE,
        }

        if action_lower in mouse_actions:
            self.click(mouse_actions[action_lower], duration)
        else:
            self.press_key(action, duration)

    def get_backend_info(self) -> dict:
        """Get information about the current input backend"""
        return {
            'backend': self._backend,
            'platform': sys.platform,
            'is_windows': self.is_windows,
        }


# Singleton instance
_default_handler = None


def get_input_handler() -> InputHandler:
    """Get the default input handler instance"""
    global _default_handler
    if _default_handler is None:
        _default_handler = InputHandler()
    return _default_handler
