"""
Humanizer Module - Creates human-like timing patterns for key/mouse actions

Uses Gaussian distribution, micro-jitter, fatigue simulation, and random
micro-pauses to make automated inputs appear natural and avoid detection.

All parameters are built-in optimized defaults - no user configuration needed.
"""

import random
import time
from dataclasses import dataclass
from typing import Tuple


@dataclass
class HumanizerConfig:
    """Built-in optimized configuration for human-like behavior"""
    # Base timing (in seconds)
    base_delay: float = 0.050          # 50ms base delay (~20 actions/sec)
    gaussian_stddev: float = 0.015     # 15ms standard deviation

    # Key press duration (humans don't press/release instantly)
    min_press_duration: float = 0.010  # 10ms minimum hold time
    max_press_duration: float = 0.040  # 40ms maximum hold time

    # Micro-jitter
    jitter_range: float = 0.005        # ±5ms random jitter

    # Fatigue simulation
    fatigue_increment: float = 0.0001  # 0.01% slowdown per 1000 actions
    fatigue_threshold: int = 1000      # Actions before fatigue kicks in
    max_fatigue_factor: float = 1.3    # Maximum 30% slowdown

    # Random micro-pauses
    micro_pause_chance: float = 0.01   # 1% chance of micro-pause
    micro_pause_min: float = 0.050     # 50ms minimum pause
    micro_pause_max: float = 0.150     # 150ms maximum pause

    # Minimum delay (safety floor)
    min_delay: float = 0.001           # 1ms minimum


class Humanizer:
    """
    Generates human-like timing delays for automated input actions.

    Uses statistical randomization to mimic natural human behavior:
    - Gaussian distribution for main delay (not uniform random)
    - Variable key press durations
    - Micro-jitter for extra randomness
    - Fatigue simulation over extended use
    - Occasional random pauses
    """

    def __init__(self, config: HumanizerConfig = None):
        """Initialize the humanizer with optional custom config"""
        self.config = config or HumanizerConfig()
        self.action_count = 0
        self.fatigue_factor = 1.0
        self._reset_session()

    def _reset_session(self):
        """Reset session statistics"""
        self.action_count = 0
        self.fatigue_factor = 1.0
        self.session_start = time.time()

    def reset(self):
        """Public method to reset the humanizer state"""
        self._reset_session()

    def _update_fatigue(self):
        """Update fatigue factor based on action count"""
        if self.action_count > 0 and self.action_count % self.config.fatigue_threshold == 0:
            self.fatigue_factor = min(
                self.fatigue_factor + self.config.fatigue_increment,
                self.config.max_fatigue_factor
            )

    def get_delay(self) -> float:
        """
        Generate a human-like delay between actions.

        Returns:
            float: Delay in seconds
        """
        self.action_count += 1
        self._update_fatigue()

        # Gaussian distribution centered on base delay
        delay = random.gauss(self.config.base_delay, self.config.gaussian_stddev)

        # Add micro-jitter
        delay += random.uniform(-self.config.jitter_range, self.config.jitter_range)

        # Apply fatigue factor
        delay *= self.fatigue_factor

        # Random micro-pause (1% chance)
        if random.random() < self.config.micro_pause_chance:
            delay += random.uniform(
                self.config.micro_pause_min,
                self.config.micro_pause_max
            )

        # Ensure minimum delay
        return max(self.config.min_delay, delay)

    def get_press_duration(self) -> float:
        """
        Generate a human-like key press duration.

        Humans don't press and release keys instantly - there's a natural
        hold time that varies slightly each press.

        Returns:
            float: Press duration in seconds
        """
        # Use triangular distribution - more likely to be in the middle
        return random.triangular(
            self.config.min_press_duration,
            self.config.max_press_duration,
            (self.config.min_press_duration + self.config.max_press_duration) / 2
        )

    def get_delay_and_duration(self) -> Tuple[float, float]:
        """
        Get both delay and press duration in one call.

        Returns:
            Tuple[float, float]: (delay_before_action, press_duration)
        """
        return self.get_delay(), self.get_press_duration()

    def get_stats(self) -> dict:
        """
        Get current humanizer statistics.

        Returns:
            dict: Statistics including action count, fatigue, session duration
        """
        return {
            "action_count": self.action_count,
            "fatigue_factor": self.fatigue_factor,
            "session_duration": time.time() - self.session_start,
            "estimated_cps": 1.0 / (self.config.base_delay * self.fatigue_factor)
        }


# Pre-generate delay batches for performance optimization
class DelayBatchGenerator:
    """
    Pre-generates batches of random delays to reduce computation during execution.

    Useful for high-frequency spamming where generating random numbers
    on every iteration could introduce overhead.
    """

    def __init__(self, humanizer: Humanizer, batch_size: int = 1000):
        self.humanizer = humanizer
        self.batch_size = batch_size
        self.delays = []
        self.durations = []
        self.index = 0
        self._generate_batch()

    def _generate_batch(self):
        """Generate a new batch of delays and durations"""
        self.delays = [self.humanizer.get_delay() for _ in range(self.batch_size)]
        self.durations = [self.humanizer.get_press_duration() for _ in range(self.batch_size)]
        self.index = 0

    def get_next(self) -> Tuple[float, float]:
        """
        Get the next pre-generated delay and duration.

        Returns:
            Tuple[float, float]: (delay, duration)
        """
        if self.index >= self.batch_size:
            self._generate_batch()

        delay = self.delays[self.index]
        duration = self.durations[self.index]
        self.index += 1
        return delay, duration


# Singleton instance for easy access
_default_humanizer = None


def get_humanizer() -> Humanizer:
    """Get the default humanizer instance (singleton)"""
    global _default_humanizer
    if _default_humanizer is None:
        _default_humanizer = Humanizer()
    return _default_humanizer


def get_delay() -> float:
    """Convenience function to get a delay from the default humanizer"""
    return get_humanizer().get_delay()


def get_press_duration() -> float:
    """Convenience function to get a press duration from the default humanizer"""
    return get_humanizer().get_press_duration()
