"""
Hotkey Capture Manager - Enhanced hotkey system with capture mode

Supports 4 hotkey types:
- Toggle: Press to start/stop
- Hold: Hold to activate, release to deactivate
- Focus Hold: Hold to spam only a specific key
- Right-Click Hold: Hold to spam only right-click

Features:
- Capture mode for setting hotkeys by pressing keys
- Both keyboard and mouse button support
- Thread-safe with proper locking
- Callback system for events
"""

import threading
from typing import Callable, Dict, List, Optional, Any
from dataclasses import dataclass, field
from pynput import keyboard, mouse

from .captured_key import CapturedKey


@dataclass
class HotkeyState:
    """State for a single hotkey"""
    key: Optional[CapturedKey] = None
    enabled: bool = True
    active: bool = False  # Currently pressed (for hold types)


class HotkeyCaptureManager:
    """
    Enhanced hotkey manager with capture mode and multiple hotkey types.

    Thread-safe implementation using pynput listeners for both
    keyboard and mouse button detection.
    """

    # Hotkey type constants
    TOGGLE = "toggle"
    HOLD = "hold"
    FOCUS_HOLD = "focus_hold"
    RIGHT_CLICK_HOLD = "right_click_hold"

    ALL_TYPES = [TOGGLE, HOLD, FOCUS_HOLD, RIGHT_CLICK_HOLD]

    def __init__(self):
        """Initialize the hotkey capture manager"""
        self._lock = threading.RLock()

        # Hotkey states
        self._hotkeys: Dict[str, HotkeyState] = {
            self.TOGGLE: HotkeyState(),
            self.HOLD: HotkeyState(enabled=False),
            self.FOCUS_HOLD: HotkeyState(enabled=False),
            self.RIGHT_CLICK_HOLD: HotkeyState(enabled=False),
        }

        # Capture mode state
        self._capturing = False
        self._capture_target: Optional[str] = None

        # Callbacks
        self._toggle_callbacks: List[Callable[[], None]] = []
        self._hold_start_callbacks: List[Callable[[], None]] = []
        self._hold_end_callbacks: List[Callable[[], None]] = []
        self._focus_hold_start_callbacks: List[Callable[[CapturedKey], None]] = []
        self._focus_hold_end_callbacks: List[Callable[[], None]] = []
        self._right_click_hold_start_callbacks: List[Callable[[], None]] = []
        self._right_click_hold_end_callbacks: List[Callable[[], None]] = []
        self._capture_callbacks: List[Callable[[str, CapturedKey], None]] = []

        # Listeners
        self._keyboard_listener: Optional[keyboard.Listener] = None
        self._mouse_listener: Optional[mouse.Listener] = None
        self._running = False

    # =========================================================================
    # Callback Registration
    # =========================================================================

    def on_toggle(self, callback: Callable[[], None]):
        """Register callback for toggle hotkey activation"""
        with self._lock:
            self._toggle_callbacks.append(callback)

    def on_hold_start(self, callback: Callable[[], None]):
        """Register callback for hold hotkey press"""
        with self._lock:
            self._hold_start_callbacks.append(callback)

    def on_hold_end(self, callback: Callable[[], None]):
        """Register callback for hold hotkey release"""
        with self._lock:
            self._hold_end_callbacks.append(callback)

    def on_focus_hold_start(self, callback: Callable[[CapturedKey], None]):
        """Register callback for focus hold press (receives the focus key)"""
        with self._lock:
            self._focus_hold_start_callbacks.append(callback)

    def on_focus_hold_end(self, callback: Callable[[], None]):
        """Register callback for focus hold release"""
        with self._lock:
            self._focus_hold_end_callbacks.append(callback)

    def on_right_click_hold_start(self, callback: Callable[[], None]):
        """Register callback for right-click hold press"""
        with self._lock:
            self._right_click_hold_start_callbacks.append(callback)

    def on_right_click_hold_end(self, callback: Callable[[], None]):
        """Register callback for right-click hold release"""
        with self._lock:
            self._right_click_hold_end_callbacks.append(callback)

    def on_key_captured(self, callback: Callable[[str, CapturedKey], None]):
        """Register callback for when a key is captured (target, key)"""
        with self._lock:
            self._capture_callbacks.append(callback)

    # =========================================================================
    # Hotkey Configuration
    # =========================================================================

    def set_hotkey(self, hotkey_type: str, key: Optional[CapturedKey]):
        """Set the key for a hotkey type"""
        with self._lock:
            if hotkey_type in self._hotkeys:
                self._hotkeys[hotkey_type].key = key

    def get_hotkey(self, hotkey_type: str) -> Optional[CapturedKey]:
        """Get the current key for a hotkey type"""
        with self._lock:
            if hotkey_type in self._hotkeys:
                return self._hotkeys[hotkey_type].key
            return None

    def set_enabled(self, hotkey_type: str, enabled: bool):
        """Enable/disable a hotkey type"""
        with self._lock:
            if hotkey_type in self._hotkeys:
                self._hotkeys[hotkey_type].enabled = enabled

    def is_enabled(self, hotkey_type: str) -> bool:
        """Check if a hotkey type is enabled"""
        with self._lock:
            if hotkey_type in self._hotkeys:
                return self._hotkeys[hotkey_type].enabled
            return False

    def is_active(self, hotkey_type: str) -> bool:
        """Check if a hold hotkey is currently active (pressed)"""
        with self._lock:
            if hotkey_type in self._hotkeys:
                return self._hotkeys[hotkey_type].active
            return False

    # =========================================================================
    # Capture Mode
    # =========================================================================

    def start_capture(self, target: str):
        """
        Start capture mode for a specific hotkey type.

        The next key or mouse button pressed will be captured.

        Args:
            target: The hotkey type to capture for ("toggle", "hold", etc.)
        """
        with self._lock:
            if target not in self._hotkeys:
                return
            self._capturing = True
            self._capture_target = target
            print(f"[HotkeyCaptureManager] Capture mode started for: {target}")

    def cancel_capture(self):
        """Cancel capture mode without setting a key"""
        with self._lock:
            self._capturing = False
            self._capture_target = None
            print("[HotkeyCaptureManager] Capture mode cancelled")

    def is_capturing(self) -> bool:
        """Check if currently in capture mode"""
        with self._lock:
            return self._capturing

    def get_capture_target(self) -> Optional[str]:
        """Get the current capture target"""
        with self._lock:
            return self._capture_target

    def _handle_capture(self, captured_key: CapturedKey):
        """Handle a captured key during capture mode"""
        target = None
        callbacks = []

        with self._lock:
            if not self._capturing:
                return

            target = self._capture_target
            self._hotkeys[target].key = captured_key
            self._capturing = False
            self._capture_target = None
            callbacks = self._capture_callbacks.copy()

        print(f"[HotkeyCaptureManager] Captured {captured_key} for {target}")

        # Fire callbacks outside lock
        for callback in callbacks:
            try:
                callback(target, captured_key)
            except Exception as e:
                print(f"[HotkeyCaptureManager] Error in capture callback: {e}")

    # =========================================================================
    # Event Handlers
    # =========================================================================

    def _on_key_press(self, key):
        """Handle keyboard key press"""
        captured_key = CapturedKey.from_pynput_key(key)

        # Check capture mode first
        with self._lock:
            if self._capturing:
                # Handle capture outside lock
                pass
            else:
                # Check against registered hotkeys
                self._check_hotkey_press(captured_key)
                return

        # Handle capture (outside lock)
        self._handle_capture(captured_key)

    def _on_key_release(self, key):
        """Handle keyboard key release"""
        captured_key = CapturedKey.from_pynput_key(key)
        self._check_hotkey_release(captured_key)

    def _on_mouse_click(self, x, y, button, pressed):
        """Handle mouse button click"""
        captured_key = CapturedKey.from_mouse_button(button)

        if pressed:
            # Check capture mode first
            with self._lock:
                if self._capturing:
                    pass  # Handle outside lock
                else:
                    self._check_hotkey_press(captured_key)
                    return

            # Handle capture (outside lock)
            self._handle_capture(captured_key)
        else:
            self._check_hotkey_release(captured_key)

    def _check_hotkey_press(self, pressed_key: CapturedKey):
        """Check if pressed key matches any registered hotkey and fire callbacks"""
        callbacks_to_fire = []

        with self._lock:
            # Toggle hotkey
            toggle_state = self._hotkeys[self.TOGGLE]
            if toggle_state.key and toggle_state.key.matches(pressed_key.key):
                callbacks_to_fire.append(('toggle', self._toggle_callbacks.copy()))

            # Hold hotkey
            hold_state = self._hotkeys[self.HOLD]
            if (hold_state.enabled and hold_state.key and
                hold_state.key.matches(pressed_key.key) and not hold_state.active):
                hold_state.active = True
                callbacks_to_fire.append(('hold_start', self._hold_start_callbacks.copy()))

            # Focus hold hotkey
            focus_state = self._hotkeys[self.FOCUS_HOLD]
            if (focus_state.enabled and focus_state.key and
                focus_state.key.matches(pressed_key.key) and not focus_state.active):
                focus_state.active = True
                # Pass the focus key to callbacks
                callbacks_to_fire.append(('focus_start', self._focus_hold_start_callbacks.copy(), focus_state.key))

            # Right-click hold hotkey
            rclick_state = self._hotkeys[self.RIGHT_CLICK_HOLD]
            if (rclick_state.enabled and rclick_state.key and
                rclick_state.key.matches(pressed_key.key) and not rclick_state.active):
                rclick_state.active = True
                callbacks_to_fire.append(('rclick_start', self._right_click_hold_start_callbacks.copy()))

        # Fire callbacks outside lock
        for item in callbacks_to_fire:
            if item[0] == 'toggle':
                for cb in item[1]:
                    try:
                        cb()
                    except Exception as e:
                        print(f"[HotkeyCaptureManager] Error in toggle callback: {e}")
            elif item[0] == 'hold_start':
                for cb in item[1]:
                    try:
                        cb()
                    except Exception as e:
                        print(f"[HotkeyCaptureManager] Error in hold_start callback: {e}")
            elif item[0] == 'focus_start':
                for cb in item[1]:
                    try:
                        cb(item[2])  # Pass the focus key
                    except Exception as e:
                        print(f"[HotkeyCaptureManager] Error in focus_start callback: {e}")
            elif item[0] == 'rclick_start':
                for cb in item[1]:
                    try:
                        cb()
                    except Exception as e:
                        print(f"[HotkeyCaptureManager] Error in rclick_start callback: {e}")

    def _check_hotkey_release(self, released_key: CapturedKey):
        """Check if released key matches any active hold hotkey"""
        callbacks_to_fire = []

        with self._lock:
            # Hold hotkey
            hold_state = self._hotkeys[self.HOLD]
            if (hold_state.active and hold_state.key and
                hold_state.key.matches(released_key.key)):
                hold_state.active = False
                callbacks_to_fire.append(('hold_end', self._hold_end_callbacks.copy()))

            # Focus hold hotkey
            focus_state = self._hotkeys[self.FOCUS_HOLD]
            if (focus_state.active and focus_state.key and
                focus_state.key.matches(released_key.key)):
                focus_state.active = False
                callbacks_to_fire.append(('focus_end', self._focus_hold_end_callbacks.copy()))

            # Right-click hold hotkey
            rclick_state = self._hotkeys[self.RIGHT_CLICK_HOLD]
            if (rclick_state.active and rclick_state.key and
                rclick_state.key.matches(released_key.key)):
                rclick_state.active = False
                callbacks_to_fire.append(('rclick_end', self._right_click_hold_end_callbacks.copy()))

        # Fire callbacks outside lock
        for item in callbacks_to_fire:
            for cb in item[1]:
                try:
                    cb()
                except Exception as e:
                    print(f"[HotkeyCaptureManager] Error in {item[0]} callback: {e}")

    # =========================================================================
    # Listener Control
    # =========================================================================

    def start(self):
        """Start the hotkey listeners"""
        if self._running:
            return

        self._running = True

        # Start keyboard listener
        self._keyboard_listener = keyboard.Listener(
            on_press=self._on_key_press,
            on_release=self._on_key_release
        )
        self._keyboard_listener.start()

        # Start mouse listener
        self._mouse_listener = mouse.Listener(
            on_click=self._on_mouse_click
        )
        self._mouse_listener.start()

        print("[HotkeyCaptureManager] Listeners started")

    def stop(self):
        """Stop the hotkey listeners"""
        if not self._running:
            return

        self._running = False

        if self._keyboard_listener:
            self._keyboard_listener.stop()
            self._keyboard_listener = None

        if self._mouse_listener:
            self._mouse_listener.stop()
            self._mouse_listener = None

        # Reset all active states
        with self._lock:
            for state in self._hotkeys.values():
                state.active = False

        print("[HotkeyCaptureManager] Listeners stopped")

    def is_running(self) -> bool:
        """Check if listeners are running"""
        return self._running

    # =========================================================================
    # Configuration Export/Import
    # =========================================================================

    def get_config(self) -> dict:
        """Export current configuration as dictionary"""
        with self._lock:
            config = {}
            for hotkey_type, state in self._hotkeys.items():
                config[hotkey_type] = {
                    "key": state.key.to_dict() if state.key else None,
                    "enabled": state.enabled,
                }
            return config

    def set_config(self, config: dict):
        """Import configuration from dictionary"""
        with self._lock:
            for hotkey_type, data in config.items():
                if hotkey_type in self._hotkeys:
                    state = self._hotkeys[hotkey_type]
                    if data.get("key"):
                        state.key = CapturedKey.from_dict(data["key"])
                    else:
                        state.key = None
                    state.enabled = data.get("enabled", True)

    def get_status(self) -> dict:
        """Get current status information"""
        with self._lock:
            return {
                "running": self._running,
                "capturing": self._capturing,
                "capture_target": self._capture_target,
                "hotkeys": {
                    htype: {
                        "key": str(state.key) if state.key else None,
                        "enabled": state.enabled,
                        "active": state.active,
                    }
                    for htype, state in self._hotkeys.items()
                }
            }


# Singleton instance
_default_manager: Optional[HotkeyCaptureManager] = None


def get_hotkey_capture_manager() -> HotkeyCaptureManager:
    """Get the default hotkey capture manager instance"""
    global _default_manager
    if _default_manager is None:
        _default_manager = HotkeyCaptureManager()
    return _default_manager
