"""
CapturedKey Module - Dataclass for storing captured hotkeys

Supports both keyboard keys (pynput Key/KeyCode) and mouse buttons (pynput Button).
Provides factory methods for creation and serialization for config storage.
"""

from dataclasses import dataclass
from typing import Optional, Union, Any
from pynput.keyboard import Key, KeyCode
from pynput.mouse import Button


@dataclass
class CapturedKey:
    """
    Stores a captured key or mouse button for hotkey configuration.

    Attributes:
        key: The pynput Key, KeyCode, or Button object
        name: Human-readable display name (e.g., "F6", "MOUSE_LEFT")
        is_mouse: True if this is a mouse button, False for keyboard
    """
    key: Union[Key, KeyCode, Button, None]
    name: str
    is_mouse: bool = False

    # Mapping of pynput Key enum to display names
    KEY_NAMES = {
        Key.f1: "F1", Key.f2: "F2", Key.f3: "F3", Key.f4: "F4",
        Key.f5: "F5", Key.f6: "F6", Key.f7: "F7", Key.f8: "F8",
        Key.f9: "F9", Key.f10: "F10", Key.f11: "F11", Key.f12: "F12",
        Key.space: "SPACE", Key.enter: "ENTER", Key.tab: "TAB",
        Key.backspace: "BACKSPACE", Key.delete: "DELETE",
        Key.esc: "ESC", Key.caps_lock: "CAPS",
        Key.shift: "SHIFT", Key.shift_l: "L-SHIFT", Key.shift_r: "R-SHIFT",
        Key.ctrl: "CTRL", Key.ctrl_l: "L-CTRL", Key.ctrl_r: "R-CTRL",
        Key.alt: "ALT", Key.alt_l: "L-ALT", Key.alt_r: "R-ALT",
        Key.cmd: "WIN", Key.cmd_l: "L-WIN", Key.cmd_r: "R-WIN",
        Key.up: "UP", Key.down: "DOWN", Key.left: "LEFT", Key.right: "RIGHT",
        Key.home: "HOME", Key.end: "END",
        Key.page_up: "PGUP", Key.page_down: "PGDN",
        Key.insert: "INSERT", Key.num_lock: "NUMLOCK",
        Key.scroll_lock: "SCRLOCK", Key.menu: "MENU",
    }

    # Reverse mapping for recreating keys from names
    NAME_TO_KEY = {v: k for k, v in KEY_NAMES.items()}

    # Mouse button mappings
    BUTTON_NAMES = {
        Button.left: "MOUSE_LEFT",
        Button.right: "MOUSE_RIGHT",
        Button.middle: "MOUSE_MIDDLE",
    }

    # Add x1/x2 buttons if available (platform dependent)
    try:
        BUTTON_NAMES[Button.x1] = "MOUSE_4"
        BUTTON_NAMES[Button.x2] = "MOUSE_5"
    except AttributeError:
        pass  # x1/x2 not available on this platform

    NAME_TO_BUTTON = {v: k for k, v in BUTTON_NAMES.items()}

    @classmethod
    def from_pynput_key(cls, key: Union[Key, KeyCode]) -> 'CapturedKey':
        """
        Create a CapturedKey from a pynput keyboard key.

        Args:
            key: pynput Key enum or KeyCode

        Returns:
            CapturedKey instance
        """
        if isinstance(key, Key):
            # Special key (F1-F12, Enter, etc.)
            name = cls.KEY_NAMES.get(key, key.name.upper() if hasattr(key, 'name') else str(key))
        elif isinstance(key, KeyCode):
            # Regular character key
            if key.char:
                name = key.char.upper()
            elif key.vk:
                # Virtual key code (for keys without char representation)
                name = f"VK_{key.vk}"
            else:
                name = str(key)
        else:
            name = str(key)

        return cls(key=key, name=name, is_mouse=False)

    @classmethod
    def from_mouse_button(cls, button: Button) -> 'CapturedKey':
        """
        Create a CapturedKey from a pynput mouse button.

        Args:
            button: pynput Button enum

        Returns:
            CapturedKey instance
        """
        name = cls.BUTTON_NAMES.get(button, str(button).replace("Button.", "MOUSE_").upper())
        return cls(key=button, name=name, is_mouse=True)

    @classmethod
    def from_name(cls, name: str) -> Optional['CapturedKey']:
        """
        Recreate a CapturedKey from its saved name string.

        Used when loading from config files.

        Args:
            name: The display name (e.g., "F6", "MOUSE_LEFT", "A")

        Returns:
            CapturedKey instance or None if name is invalid/empty
        """
        if not name or name == "—":
            return None

        name_upper = name.upper()

        # Check if it's a mouse button
        if name_upper in cls.NAME_TO_BUTTON:
            button = cls.NAME_TO_BUTTON[name_upper]
            return cls(key=button, name=name_upper, is_mouse=True)

        # Check if it's a special keyboard key
        if name_upper in cls.NAME_TO_KEY:
            key = cls.NAME_TO_KEY[name_upper]
            return cls(key=key, name=name_upper, is_mouse=False)

        # Check if it's a single character
        if len(name) == 1:
            return cls(key=KeyCode.from_char(name.lower()), name=name_upper, is_mouse=False)

        # Check for virtual key code format
        if name_upper.startswith("VK_"):
            try:
                vk = int(name_upper[3:])
                return cls(key=KeyCode.from_vk(vk), name=name_upper, is_mouse=False)
            except (ValueError, AttributeError):
                pass

        return None

    def matches(self, other_key: Any) -> bool:
        """
        Check if this CapturedKey matches another pynput key/button.

        Args:
            other_key: A pynput Key, KeyCode, or Button to compare against

        Returns:
            True if they represent the same key/button
        """
        if self.key is None:
            return False

        # Direct comparison
        if self.key == other_key:
            return True

        # For KeyCode, compare by char or vk
        if isinstance(self.key, KeyCode) and isinstance(other_key, KeyCode):
            if self.key.char and other_key.char:
                return self.key.char.lower() == other_key.char.lower()
            if self.key.vk and other_key.vk:
                return self.key.vk == other_key.vk

        return False

    def to_dict(self) -> dict:
        """Serialize to dictionary for config storage"""
        return {
            "name": self.name,
            "is_mouse": self.is_mouse,
        }

    @classmethod
    def from_dict(cls, data: dict) -> Optional['CapturedKey']:
        """Deserialize from dictionary"""
        if not data or not data.get("name"):
            return None
        return cls.from_name(data["name"])

    def __str__(self) -> str:
        return self.name

    def __repr__(self) -> str:
        return f"CapturedKey({self.name}, is_mouse={self.is_mouse})"

    def __eq__(self, other) -> bool:
        if isinstance(other, CapturedKey):
            return self.name == other.name and self.is_mouse == other.is_mouse
        return False

    def __hash__(self) -> int:
        return hash((self.name, self.is_mouse))
